/*!
 * Ahsay Microsoft Teams chat preview library 1.0.0
 * http://www.ahsay.com/
 *
 * Description:
 * This library provide methods to render Microsoft Teams chat messages as chat preview.
 *
 * Copyright (c) 2021
 */

const CALL_ENDED_EVENT = '#microsoft.graph.callEndedEventMessageDetail';
const CALL_RECORDING_EVENT = '#microsoft.graph.callRecordingEventMessageDetail';
const CALL_STARTED_EVENT = '#microsoft.graph.callStartedEventMessageDetail';
const CALL_TRANSCRIPT_EVENT = '#microsoft.graph.callTranscriptEventMessageDetail';
const CHANNEL_ADDED_EVENT = '#microsoft.graph.channelAddedEventMessageDetail';
const CHANNEL_DELETED_EVENT = '#microsoft.graph.channelDeletedEventMessageDetail';
const CHANNEL_DESC_UPDATED_EVENT = '#microsoft.graph.channelDescriptionUpdatedEventMessageDetail';
const CHANNEL_RENAMED_EVENT = '#microsoft.graph.channelRenamedEventMessageDetail';
const CHANNEL_SET_AS_FAV_EVENT = '#microsoft.graph.channelSetAsFavoriteByDefaultEventMessageDetail';
const CHANNEL_UNSET_AS_FAV_EVENT = '#microsoft.graph.channelUnsetAsFavoriteByDefaultEventMessageDetail'
const CHAT_RENAMED_EVENT = '#microsoft.graph.chatRenamedEventMessageDetail';
const MEMBER_ROLE_UPDATED_EVENT = '#microsoft.graph.conversationMemberRoleUpdatedEventMessageDetail';
const MEETING_POLICY_UPDATED_EVENT = '#microsoft.graph.meetingPolicyUpdatedEventMessageDetail';
const MEMBERS_ADDED_EVENT = '#microsoft.graph.membersAddedEventMessageDetail';
const MEMBERS_DELETED_EVENT = '#microsoft.graph.membersDeletedEventMessageDetail';
const MEMBERS_JOINED_EVENT = '#microsoft.graph.membersJoinedEventMessageDetail';
const MEMBERS_LEFT_EVENT = '#microsoft.graph.membersLeftEventMessageDetail';
const TAB_UPDATED_EVENT = '#microsoft.graph.tabUpdatedEventMessageDetail';
const TEAM_ARCHIVED_EVENT = '#microsoft.graph.teamArchivedEventMessageDetail';
const TEAM_CREATED_EVENT = '#microsoft.graph.teamCreatedEventMessageDetail';
const TEAM_DESC_UPDATED_EVENT = '#microsoft.graph.teamDescriptionUpdatedEventMessageDetail';
const TEAM_JOINING_DISABLED_EVENT = '#microsoft.graph.teamJoiningDisabledEventMessageDetail';
const TEAM_JOINING_ENABLED_EVENT = '#microsoft.graph.teamJoiningEnabledEventMessageDetail';
const TEAM_RENAMED_EVENT = '#microsoft.graph.teamRenamedEventMessageDetail';
const TEAMS_APP_INSTALLED_EVENT = '#microsoft.graph.teamsAppInstalledEventMessageDetail';
const TEAMS_APP_REMOVED_EVENT = '#microsoft.graph.teamsAppRemovedEventMessageDetail';
const TEAMS_APP_UPGRADED_EVENT = '#microsoft.graph.teamsAppUpgradedEventMessageDetail';
const TEAM_UNARCHIVED_EVENT = '#microsoft.graph.teamUnarchivedEventMessageDetail';
const SHARED_ALL_HISTORY = '0001-01-01T00:00:00Z';

// Settings and data
const DATE_OPTIONS = { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' };
const TIME_OPTIONS = { hour: '2-digit', minute: '2-digit' };
const BATCH_COUNT = 20;
const SCROLL_THEADHOLD = 0.2 * _getHeight();
var pageNum = 0;
var prevDate = null;
var CHANNEL = false;
var EXPORT = false;
var DISPLAY_NAME_MAP = {};

/*
 * Java connector
 */
var jsConnector = {
    changeLanguage: function(locale, langPackStr) {
        changeLanguage(locale, langPackStr);
    },
    initMessages: function() {
        _loadNextPage();
    },
    loadMoreMessages: function(messages) {
        if (messages) {
            loadMoreMessages(messages);
        }
    },
    loadReplyMessages: function(messages) {
        if (messages) {
            loadReplyMessages(messages);
        }
    },
    loadHostedContents: function(hostedContents) {
        if (hostedContents) {
            loadHostedContents(hostedContents);
        }
    },
    loadAttachments: function(attachments) {
        if (attachments) {
            loadAttachments(attachments);
        }
    },
    showChat: function(user, chat, members) {
        showChat(user, chat, members);
    },
    showChannel: function(channel, members) {
        showChannel(channel, members);
    }
}

function getJsConnector() {
    return jsConnector;
}

function getJavaConnector() {
    if (!window.javaConnector) {
        console.log('Java connector not defined!');
    } else {
        return window.javaConnector;
    }
}

$(document).ready(function() {
    let teamsBody = $('#teams-body');

    EXPORT = isExport();

    if (EXPORT) {
        // Render preload messages in export
        CHANNEL = isChannel();
        changeLanguage(LOCALE, null);
        _preloadProperties();
        _preloadMessages();
    } else {
        if (isCBS()) {
            teamsPreviewReady();
        } else {
            teamsBody.scroll(function(event) {
                let scroll = teamsBody.scrollTop();
                if (scroll <= SCROLL_THEADHOLD) {
                    // Lazy loading
                    _loadNextPage();
                }
            });
            _loadNextPage();
        }
    }
    // Scroll to bottom initially
    teamsBody.scrollTop(teamsBody.prop('scrollHeight'));
});

/* Public methods begin */
function changeLanguage(locale, langPackStr) {
    console.log('Changing locale: ' + locale);
    console.log('Supported locales: ' + navigator.languages);
    if (locale) {
        console.log('Using locale: ' + locale);
        LOCALE = locale;
    }
    if (langPackStr) {
        var langPack;
        try {
            langPack = JSON.parse(langPackStr);
        } catch (error) {
            console.log(error);
        }
        if (langPack) {
            LANG_LIKE = langPack.LANG_LIKE;
            LANG_HEART = langPack.LANG_HEART;
            LANG_LAUGH = langPack.LANG_LAUGH;
            LANG_SURPRISED = langPack.LANG_SURPRISED;
            LANG_SAD = langPack.LANG_SAD;
            LANG_ANGRY = langPack.LANG_ANGRY;
            LANG_REACTION = langPack.LANG_REACTION;
            LANG_REACTIONS = langPack.LANG_REACTIONS;
            LANG_EDITED = langPack.LANG_EDITED;
            LANG_DELETED = langPack.LANG_DELETED;
            LANG_URGENT = langPack.LANG_URGENT;
            LANG_IMPORTANT = langPack.LANG_IMPORTANT;
        }
    }
}

function showChat(user, chat, members) {
    _clearAllMessages();
    _setChatProperties(user, chat, members);
}

function showChannel(channel, members) {
    _clearAllMessages();
    _setChannelProperties(channel, members);
}

function loadAttachments(attachmentsStr) {
    var attachments;
    try {
        attachments = JSON.parse(attachmentsStr);
    } catch (error) {
        console.log(error);
    }
    if (attachments) {
        _insertAttachments(attachments);
    }
}

function isChannel(object) {
    return (typeof channelProperties !== 'undefined' && channelProperties != null);
}

function isExport(object) {
    return (typeof masterMessageList !== 'undefined' && masterMessageList != null);
}

function isCBS() {
    return (typeof CBS !== 'undefined' && CBS);
}

function loadMoreMessages(messagesStr) {
    if (messagesStr !== '') {
        try {
            let messages = JSON.parse(messagesStr);
            if (messages && messages.length > 0) {
                let teamsBody = $('#teams-body');
                let prevHeight = teamsBody.prop('scrollHeight');
                _insertMessages(messages);
                // Retain scroll position
                teamsBody.scrollTop(teamsBody.prop('scrollHeight') - prevHeight);
                pageNum++;
                if (messages.length < BATCH_COUNT && prevDate) {
                    // Insert the last date line
                    _insertDateLine(prevDate);
                    prevDate = null;
                }
                return;
            }
        } catch(e) {
            console.log('Invalid message format: ' + messagesStr);
        }
    }
    pageNum = -1;
    if (prevDate) {
        // Insert the last date line
        _insertDateLine(prevDate);
        prevDate = null;
    }
}

function loadReplyMessages(messagesStr) {
    var messages;
    try {
        messages = JSON.parse(messagesStr);
    } catch (error) {
        console.log(error);
    }
    if (messages) {
        _insertChannelReplyMessages(messages);
    }
}

function loadHostedContents(hostedContentsStr) {
    var hostedContents;
    try {
        hostedContents = JSON.parse(hostedContentsStr);
    } catch (error) {
        console.log(error);
    }
    if (hostedContents) {
        _insertHostedContents(hostedContents);
    }
}

function _clearAllMessages() {
    let teamsBodyDOM = document.getElementById('teams-body');
    teamsBodyDOM.innerHTML = '';
    pageNum = 0;
    prevDate = null;
    members = null;
}

function doLocalDownload(localPath) {
    localPath = encodePath(localPath);
    if (localPath) {
        window.open(localPath, '_blank');
    } else {
        console.log('[doLocalDownload] Local path not found!');
    }
}

function encodePath(localPath) {
    while (localPath.includes("%2f")) {
        localPath = localPath.replace("%2f","%252f");
    }
    return localPath;
}

function doQuickDownload(sourcePath) {
    if (sourcePath) {
        let connector = getJavaConnector();
        if (connector) {
            connector.downloadFile(sourcePath);
        } else {
            console.log('[doLocalDownload] Java connector not available!');
        }
    } else {
        console.log('[doLocalDownload] Source path not found!');
    }
}

function doPreviewMedia(sourcePath) {
    if (sourcePath) {
        let connector = getJavaConnector();
        if (connector) {
            connector.previewMedia(sourcePath);
        } else {
            console.log('[doPreviewMedia] Java connector not available!');
        }
    } else {
        console.log('[doPreviewMedia] Source path not found!');
    }
}

function doOpenUrl(url) {
    if (url) {
        let connector = getJavaConnector();
        if (connector) {
            connector.openUrl(url);
        } else {
            window.open(url, '_blank');
        }
    } else {
        console.log('[doOpenUrl] Url not found!');
    }
}

/* Private methods begin */
/**
 * Set as chatroom view with chat properties
 * Sample chat profile json
 * {
 *  "id" : "",
 *  "topic" : "",
 *  "chatType" : "",
 *  "members" : [{"id" : "", "displayName" : ""}, {"id" : "", "displayName" : ""}]
 * }
 */
function _setChatProperties(userStr, chatStr, membersStr) {
    channelProperties = null;
    CHANNEL = false;
    try {
        members = JSON.parse(membersStr);
    } catch (error) {
        console.log(error);
    }
    if (userStr) {
        try {
            me = JSON.parse(userStr);
            // Map the me ID from the member list
            if (me.id === undefined) {
                for (i = 0; i < members.length; i++) {
                    let member = members[i];
                    if (member.userPrincipalName == me.userPrincipalName) {
                        me.id = member.userId;
                        break;
                    }
                }
            }
        } catch (error) {
            console.log(error);
        }
    } else {
        me = null;
    }
    try {
        chatProperties = JSON.parse(chatStr);
        _insertHeader(chatProperties.topic);
    } catch (error) {
        console.log(error);
    }
}

/**
 * Set as channel view with channel properties
 * Sample channel profile json
 * {
 *  "id" : "",
 *  "displayName" : "",
 *  "description" : "",
 *  "membershipType" : "",
 *  "groupId" : "",
 *  "members" : [{"id" : "", "displayName" : ""}, {"id" : "", "displayName" : ""}]
 * }
 */
function _setChannelProperties(channelStr, membersStr) {
    chatProperties = null;
    CHANNEL = true;
    try {
        members = JSON.parse(membersStr);
    } catch (error) {
        console.log(error);
    }
    try {
        channelProperties = JSON.parse(channelStr);
        _insertHeader(channelProperties.displayName);
    } catch (error) {
        console.log(error);
    }
}

function _getHeight() {
    var iHeight = 0;
    if(typeof( window.innerWidth ) == 'number') {
        // Non-IE
        iHeight = window.innerHeight;
    } else if( document.documentElement && ( document.documentElement.clientWidth || document.documentElement.clientHeight ) ) {
        // IE 6+ in 'standards compliant mode'
        iHeight = document.documentElement.clientHeight;
    } else if( document.body && ( document.body.clientWidth || document.body.clientHeight ) ) {
        // IE 4 compatible
        iHeight = document.body.clientHeight;
    }
    console.log('iHeight = ' + iHeight);
    console.log('iHeight = ' + iHeight * 0.2);
    return iHeight;
}

function _preloadProperties() {
    if (CHANNEL) {
        if (channelProperties.members) {
            members = channelProperties.members;
        }
        _insertHeader(channelProperties.displayName);
    } else {
        if (chatProperties.members) {
            members = chatProperties.members;
        }
        _insertHeader(chatProperties.topic);
    }
}

/**
 * Insert the header information
 *
 *    <div class="row">
 *        <div class="img-container">
 *            <img src="./images/sample_image01.png" class="rounded-circle user-img" />
 *        </div>
 *        <div class="user-info">
 *            <span>Jacky Smith</span>
 *        </div>
 *    </div>
 *
 */
function _insertHeader(title, icon) {
    var headerDiv = "";
    if (icon !== undefined) {
        headerDiv = "<div class='img-container'><img src='" + icon + "' class='rounded-circle user-img' /></div>";
    }
    headerDiv += "<div class='user-info'><span>" + title + "</span></div>";
    $('#teams-header').html(headerDiv);
    $(document).prop('title', title);
}

function _loadNextPage() {
    if (pageNum < 0) {
        // Reached the end of chat
        return;
    }
    let iStartIdx = pageNum * BATCH_COUNT;
    let connector = getJavaConnector();
    if (connector) {
        connector.loadMoreMessages(iStartIdx, BATCH_COUNT);
    }
}

function _preloadMessages() {
    if (masterMessageList) {
        _insertMessages(masterMessageList);
    }

    // Insert reply messages
    if (CHANNEL && replyMessageList) {
        _insertChannelReplyMessages(replyMessageList);
    }
}

function _insertMessages(messageList) {
    for (let i = 0; i < messageList.length; i++) {
        let curDate = _getDateOnly(messageList[i].createdDateTime);
        if (!prevDate) {
            prevDate = curDate;
        }
        if (prevDate.getTime() != curDate.getTime()) {
            _insertDateLine(prevDate);
            prevDate = curDate;
        }
        _insertMessage(messageList[i]);
    }

    if (EXPORT) {
        if (prevDate) {
            _insertDateLine(prevDate);
        }
    }
}

function _handleSystemEvent(message, parentMsgContainer) {
    if (message.eventDetail == null) {
        return;
    }
    let rawMsg = JSON.stringify(message);
    let eventDetail = message.eventDetail;
    let eventDetailType = eventDetail['@odata.type'];
    if (eventDetailType === null) {
        return;
    }
    eventDetailType = eventDetailType.trim();
    let createdDate = new Date(message.createdDateTime);
    let formattedDate = _formatDateTime(createdDate);
    let bCall = eventDetailType === CALL_ENDED_EVENT
            || eventDetailType === CALL_STARTED_EVENT
            || eventDetailType === CALL_RECORDING_EVENT
            || eventDetailType === CALL_TRANSCRIPT_EVENT;

    let msg = _getSystemMessage(eventDetail, rawMsg, bCall, formattedDate);
    if (!msg) {
        return;
    }

    let initiator = eventDetail.initiator;
    let initiatorDisplayName = _getInitiatorDisplayName(initiator);
    if (initiator) {
        msg = msg.replace('{initiator}', initiatorDisplayName);
    }

    let participants = eventDetail.callParticipants;
    participants = participants ? participants : eventDetail.members;
    if (participants) {
        let participantsName = _concatMembersDisplayNames(participants);
        if (participantsName) {
            msg = msg.replace('{participants}', participantsName);
        }
    }

    let conversationMemberUser = eventDetail.conversationMemberUser;
    if (conversationMemberUser) {
        let conversationMemberUserName = _getDisplayName(conversationMemberUser);
        if (conversationMemberUserName) {
            msg = msg.replace('{participants}', conversationMemberUserName);
        }
    }

    let teamsBodyDOM = document.getElementById('teams-body');
    let rowDiv = document.createElement('div');
    rowDiv.className = 'd-flex';

    if (CHANNEL && bCall) {
        let msgBaseContainer = document.createElement('div');
        msgBaseContainer.id = message.id;
        let msgContainer = document.createElement('div');
        let msgBody = document.createElement('div');
        let span = document.createElement('span');
        span.className = 'channel-system-msg';
        span.innerHTML = /*'[' + formattedDate + '] ' +*/ msg;

        if (parentMsgContainer) {
        } else {
            $(rowDiv).addClass('channel-msg-container');
        }
        let msgContent = document.createElement('div');
        $(msgBaseContainer).addClass('msg-base-container');
        $(rowDiv).addClass('justify-content-start').addClass('start-container');
        _insertSystemIcon(rowDiv, eventDetailType, true);

        $(msgContainer).addClass('msg-container');
        $(msgContent).addClass('msg-content');

        if (parentMsgContainer) {
            msgContainer.className = 'msg-reply-container';
        }
        msgBody.appendChild(document.createElement('div'));
        msgContent.appendChild(span);
        msgBody.appendChild(msgContent);
        // Message body
        msgContainer.appendChild(msgBody);

        msgBaseContainer.appendChild(msgContainer);
        rowDiv.appendChild(msgBaseContainer);
        if (parentMsgContainer) {
            if (!parentMsgContainer.getElementsByClassName('msg-reply-container')) {
                parentMsgContainer.appendChild(msgContainer);
            } else {
                parentMsgContainer.insertBefore(msgContainer, parentMsgContainer.childNodes[1]);
            }
        } else if (!teamsBodyDOM.innerHTML) {
            teamsBodyDOM.appendChild(rowDiv);
        } else {
            teamsBodyDOM.insertBefore(rowDiv, teamsBodyDOM.childNodes[0]);
        }
    } else {
        rowDiv.className = 'chat-system-msg d-flex';
        rowDiv.id = message.id;

        let span = document.createElement('div');
        span.className = 'chat-system-msg-text';
        if (CHANNEL) {
            span.style = 'margin-top: 5px;'
        }
        span.innerHTML = msg;

        _insertSystemIcon(rowDiv, eventDetailType, false);
        rowDiv.appendChild(span);

        if (!teamsBodyDOM.innerHTML) {
            teamsBodyDOM.appendChild(rowDiv);
        } else {
            teamsBodyDOM.insertBefore(rowDiv, teamsBodyDOM.childNodes[0]);
        }
    }
}

function _getSystemMessage(eventDetail, rawMsg, bCall, formattedDate) {
    let eventDetailType = eventDetail['@odata.type'].trim();
    let msg = '';
    if (eventDetailType == CALL_ENDED_EVENT) {
        if (CHANNEL) {
            msg = '{callEventType} with {participants} ended: '
                + _formatCallDuration(eventDetail.callDuration);
            bCall = true;
        } else {
            msg = formattedDate + '  {callEventType} ended '
                + _formatCallDuration(eventDetail.callDuration);
        }
        msg = _setCallEventType(msg, eventDetail.callEventType);
    } else if (eventDetailType == CALL_RECORDING_EVENT) {
        msg = '{initiator} started recording.';
    } else if (eventDetailType == CALL_STARTED_EVENT) {
        if (CHANNEL) {
            msg = '{callEventType} started';
        } else {
            msg = formattedDate + '  {callEventType} started';
        }
        msg = _setCallEventType(msg, eventDetail.callEventType);
    } else if (eventDetailType == CALL_TRANSCRIPT_EVENT) {
        msg = '{initiator} callTranscriptEventMessageDetail:' + rawMsg;
    } else if (eventDetailType == CHANNEL_ADDED_EVENT) {
        // A has created channel Weekly Schedule Alignment.
        msg = '{initiator} has created channel {channelDisplayName}.';
        msg = msg.replace('{channelDisplayName}', eventDetail.channelDisplayName);
    } else if (eventDetailType == CHANNEL_DELETED_EVENT) {
        msg = '{initiator} deleted channel {channelDisplayName}.';
        msg = msg.replace('{channelDisplayName}', eventDetail.channelDisplayName);
    } else if (eventDetailType == CHANNEL_DESC_UPDATED_EVENT) {
        // {init} changed channel description.
        msg = '{initiator} changed channel description.';
    } else if (eventDetailType == CHANNEL_RENAMED_EVENT) {
        // {init} changed channel name from {old} to {new}.
        msg = '{initiator} changed channel name from {old} to {new}.';
        // No reference to old name
        msg = msg.replace('from {old} ', '');
        msg = msg.replace('{new}', eventDetail.channelDisplayName);
    } else if (eventDetailType == CHANNEL_SET_AS_FAV_EVENT) {
        // A set this channel to be automatically shown in the channels list.
        msg = '{initiator} set this channel to be automatically shown in the channels list.';
    } else if (eventDetailType == CHANNEL_UNSET_AS_FAV_EVENT) {
        // A set this channel to no longer automatically show in the channels list.
        msg = '{initiator} set this channel to no longer automatically show in the channels list';
    } else if (eventDetailType == CHAT_RENAMED_EVENT) {
        msg = '{initiator} changed the group name to {chatDisplayName}';
        msg = msg.replace('{chatDisplayName}', eventDetail.chatDisplayName);
    } else if (eventDetailType == MEMBER_ROLE_UPDATED_EVENT) {
        // A has made B a team member./owner.
        let role = eventDetail.conversationMemberRoles ? eventDetail.conversationMemberRoles[0] : '';
        if (role) {
            role = role.toLowerCase();
        }
        msg = '{initiator} has made {participants} a team ' + role + '.';
    } else if (eventDetailType == MEETING_POLICY_UPDATED_EVENT) {
        // msg = '{initiator} meetingPolicyUpdatedEventMessageDetail:' + rawMsg;
        msg = '{initiator} updated meeting policy.';
    } else if (eventDetailType == MEMBERS_ADDED_EVENT) {
        // A has added B and C to the channel.
        // msg = '{initiator} has added {participants} to the ' + (CHANNEL ? 'channel' : 'chat');
        // [8/11/2020 4:31 PM] A added D to the chat.
        // [6:23 PM] A added E to the chat and shared chat history from the past day.
        // [6:23 PM] A added E to the chat and shared all chat history.
        if (!CHANNEL) {
            /*
                let visibleHistoryStartDateTime = eventDetail.visibleHistoryStartDateTime;
                if (visibleHistoryStartDateTime) {
                    let bSharedAll = visibleHistoryStartDateTime && visibleHistoryStartDateTime == '0001-01-01T00:00:00Z';
                    msg += ' and shared ' + (bSharedAll ? 'all chat history' : 'chat history from ' + _formatDateTime(new Date(visibleHistoryStartDateTime)));
                }
            */
            msg = _getMembersAddedToChatEventMsg(eventDetail);
        } else {
            msg = _getMembersAddedToChannelEventMsg(eventDetail);
        }
    } else if (eventDetailType == MEMBERS_DELETED_EVENT) {
        // Unknown User has been removed from the team.
        // msg = '{participants} has been removed from the team.';
        if (!CHANNEL) {
            msg = _getMembersDeletedFromChatEventMsg(eventDetail);
        } else {
            msg =_getMembersDeletedFromChannelEventMsg(eventDetail);
        }
    } else if (eventDetailType == MEMBERS_JOINED_EVENT) {
        msg = '{participants} joined the meeting chat.';
    } else if (eventDetailType == MEMBERS_LEFT_EVENT) {
        msg = '{participants} left the meeting chat.';
    } else if (eventDetailType == TAB_UPDATED_EVENT) {
        // msg = '{initiator} tabUpdatedEventMessageDetail:' + rawMsg;
        msg = '{initiator} updated tab.';
    } else if (eventDetailType == TEAM_ARCHIVED_EVENT) {
        // msg = '{initiator} teamArchivedEventMessageDetail:' + rawMsg;
        msg = '{initiator} archive the team.';
    } else if (eventDetailType == TEAM_CREATED_EVENT) {
        // msg = '{initiator} teamCreatedEventMessageDetail:' + rawMsg;
        msg = '{initiator} created the team.';
    } else if (eventDetailType == TEAM_DESC_UPDATED_EVENT) {
        // Appears in General channel only
        msg = '{initiator} changed channel description.';
    } else if (eventDetailType == TEAM_JOINING_DISABLED_EVENT) {
        // No message in Teams UI
        // msg = '{initiator} teamJoiningDisabledEventMessageDetail:' + rawMsg;
    } else if (eventDetailType == TEAM_JOINING_ENABLED_EVENT) {
        // No message in Teams UI
        // msg = '{initiator} teamJoiningEnabledEventMessageDetail:' + rawMsg;
    } else if (eventDetailType == TEAM_RENAMED_EVENT) {
        // Appears in General channel only
        // {init} changed channel name from {old} to {new}.
        msg = '{initiator} changed channel name from {old} to {new}.';
        // No reference to old name
        msg = msg.replace('from {old} ', '');
        msg = msg.replace('{new}', eventDetail.teamDisplayName);
    } else if (eventDetailType == TEAMS_APP_INSTALLED_EVENT) {
        // A has added App to the team.
        msg = ' {initiator} has added {teamsAppDisplayName} to the team.';
        msg = msg.replace('{teamsAppDisplayName}', eventDetail.teamsAppDisplayName);
    } else if (eventDetailType == TEAMS_APP_REMOVED_EVENT) {
        // A removed App from the team.
        msg = '{initiator} removed {teamsAppDisplayName} from the team.';
        msg = msg.replace('{teamsAppDisplayName}', eventDetail.teamsAppDisplayName);
    } else if (eventDetailType == TEAMS_APP_UPGRADED_EVENT) {
        msg = '{initiator} upgraded the app {teamsAppDisplayName}.';
        msg = msg.replace('{teamsAppDisplayName}', eventDetail.teamsAppDisplayName);
    } else if (eventDetailType == TEAM_UNARCHIVED_EVENT) {
        // msg = '{initiator} teamUnarchivedEventMessageDetail:' + rawMsg;
        msg = '{initiator} unarchive the team.';
    }
    return msg;
}

function _insertSystemIcon(parentDiv, eventDetailType, bChannelCall) {
    if (!eventDetailType) {
        return;
    }
    let iconDiv = document.createElement('div');
    iconDiv.textContent = eventDetailType;
    if (bChannelCall) {
        const iconColor = '#5b5fc7';
        iconDiv.style.backgroundColor = iconColor;
        iconDiv.style.color = _invertColor(iconColor, true);
        $(iconDiv).addClass('circle-avatar');
        $(iconDiv).html('<svg role="presentation" viewBox="-6 -6 32 32" class="app-svg icons-call-meetup-line"><g class="icons-default-fill"><path class="icons-unfilled"  fill="#616161" d="M4.5 4C3.11929 4 2 5.11929 2 6.5V13.5C2 14.8807 3.11929 16 4.5 16H11.5C12.8807 16 14 14.8807 14 13.5V12.5L16.4 14.3C17.0592 14.7944 18 14.324 18 13.5V6.49998C18 5.67594 17.0592 5.20556 16.4 5.69998L14 7.49998V6.5C14 5.11929 12.8807 4 11.5 4H4.5ZM14 8.74998L17 6.49998V13.5L14 11.25V8.74998ZM13 6.5V13.5C13 14.3284 12.3284 15 11.5 15H4.5C3.67157 15 3 14.3284 3 13.5V6.5C3 5.67157 3.67157 5 4.5 5H11.5C12.3284 5 13 5.67157 13 6.5Z"></path><path class="icons-filled" fill="white" d="M13 6.5C13 5.11929 11.8807 4 10.5 4H4.5C3.11929 4 2 5.11929 2 6.5V13.5C2 14.8807 3.11929 16 4.5 16H10.5C11.8807 16 13 14.8807 13 13.5V6.5ZM14 7.93082V12.0815L16.7642 14.4319C17.2512 14.8461 18 14.4999 18 13.8606V6.19315C18 5.55685 17.2575 5.20962 16.7692 5.61756L14 7.93082Z"></path></g></svg>');
        parentDiv.appendChild(iconDiv);
        return;
    }

    iconDiv.className = 'system-msg-svg';
    let svg = '<svg width="35" height="35" fill="currentColor" class="bi bi-exclamation-circle" viewBox="0 0 16 16"><path d="M8 15A7 7 0 1 1 8 1a7 7 0 0 1 0 14zm0 1A8 8 0 1 0 8 0a8 8 0 0 0 0 16z"/><path d="M7.002 11a1 1 0 1 1 2 0 1 1 0 0 1-2 0zM7.1 4.995a.905.905 0 1 1 1.8 0l-.35 3.507a.552.552 0 0 1-1.1 0L7.1 4.995z"/></svg>';
    switch (eventDetailType) {
        case CALL_ENDED_EVENT:
            svg = '<svg viewBox="2 2 16 16" role="presentation" focusable="false" class="fm fn fo fp fq"><path class="ui-icon__outline fr" d="M6.986 2.066l-.717.216a3.5 3.5 0 00-2.454 2.854c-.297 2.068.367 4.486 1.968 7.258 1.597 2.767 3.355 4.55 5.29 5.329a3.5 3.5 0 003.715-.705l.543-.514a2 2 0 00.246-2.623l-1.357-1.88a1.5 1.5 0 00-1.655-.556l-2.05.627-.053.01c-.226.033-.748-.456-1.398-1.582-.68-1.178-.82-1.867-.633-2.045l1.043-.973a2.497 2.497 0 00.575-2.85l-.662-1.471a2 2 0 00-2.4-1.095zm1.49 1.505l.66 1.471a1.497 1.497 0 01-.344 1.71l-1.046.974C7.077 8.36 7.3 9.442 8.198 11c.846 1.466 1.618 2.19 2.448 2.064l.124-.026 2.088-.637a.5.5 0 01.552.185l1.356 1.88a1 1 0 01-.123 1.312l-.543.514a2.5 2.5 0 01-2.653.503c-1.698-.684-3.303-2.311-4.798-4.9C5.15 9.3 4.544 7.091 4.805 5.277a2.5 2.5 0 011.753-2.039l.717-.216a1 1 0 011.2.548z"></path></svg>';
            iconDiv.className = 'system-msg-smaller-svg';
            break;
        case CALL_STARTED_EVENT:
            iconDiv.className = 'system-msg-smaller-svg';
            svg = '<svg viewBox="2 2 16 16" role="presentation" focusable="false" class="fm fn fo fp fq"><path class="ui-icon__outline fr" d="M6.986 2.066l-.717.216a3.5 3.5 0 00-2.454 2.854c-.297 2.068.367 4.486 1.968 7.258 1.597 2.767 3.355 4.55 5.29 5.329a3.5 3.5 0 003.715-.705l.543-.514a2 2 0 00.246-2.623l-1.357-1.88a1.5 1.5 0 00-1.655-.556l-2.05.627-.053.01c-.226.033-.748-.456-1.398-1.582-.68-1.178-.82-1.867-.633-2.045l1.043-.973a2.497 2.497 0 00.575-2.85l-.662-1.471a2 2 0 00-2.4-1.095zm1.49 1.505l.66 1.471a1.497 1.497 0 01-.344 1.71l-1.046.974C7.077 8.36 7.3 9.442 8.198 11c.846 1.466 1.618 2.19 2.448 2.064l.124-.026 2.088-.637a.5.5 0 01.552.185l1.356 1.88a1 1 0 01-.123 1.312l-.543.514a2.5 2.5 0 01-2.653.503c-1.698-.684-3.303-2.311-4.798-4.9C5.15 9.3 4.544 7.091 4.805 5.277a2.5 2.5 0 011.753-2.039l.717-.216a1 1 0 011.2.548z"></path><path style="fill:#5b5fc7;" class="ui-icon__filled" d="M6.986 2.066a2 2 0 012.327.946l.074.149.662 1.471a2.497 2.497 0 01-.442 2.718l-.133.132-1.043.973c-.188.178-.047.867.633 2.045.612 1.06 1.11 1.555 1.355 1.582h.043l.053-.01 2.05-.627a1.5 1.5 0 011.564.441l.091.115 1.357 1.88a2 2 0 01-.125 2.497l-.121.126-.543.514a3.5 3.5 0 01-3.715.705c-1.935-.78-3.693-2.562-5.29-5.329-1.6-2.772-2.265-5.19-1.968-7.258a3.5 3.5 0 012.262-2.79l.192-.064.717-.216z"></path></svg>';
            break;
        case CHAT_RENAMED_EVENT:
            iconDiv.className = 'system-msg-smaller-svg';
            svg = '<svg role="presentation" focusable="false" viewBox="2 2 16 16" class="fm fn fo fp fq"><g><path class="ui-icon__outline fm" d="M13.2452 2.81739C14.332 1.73059 16.0941 1.73059 17.1809 2.81739C18.2224 3.85892 18.2658 5.5206 17.3111 6.6138L17.1809 6.75307L7.57506 16.3589C7.36997 16.564 7.12323 16.7215 6.85236 16.8213L6.68708 16.8742L2.63211 17.9801C2.28552 18.0746 1.96546 17.7861 2.00301 17.4455L2.01817 17.3662L3.12407 13.3112C3.20039 13.0314 3.33646 12.7722 3.52212 12.5511L3.63939 12.4232L13.2452 2.81739ZM12.385 5.09229L4.34649 13.1303C4.2542 13.2226 4.18117 13.3318 4.13111 13.4514L4.08884 13.5743L3.2122 16.785L6.42397 15.9094C6.50791 15.8865 6.58815 15.8529 6.66278 15.8096L6.77028 15.7375L6.86796 15.6518L14.906 7.61329L12.385 5.09229ZM16.4738 3.5245C15.8162 2.8669 14.7727 2.83037 14.0722 3.4149L13.9523 3.5245L13.092 4.38529L15.613 6.90629L16.4738 6.04596C17.1314 5.38836 17.1679 4.34488 16.5834 3.64441L16.4738 3.5245Z"></path></g></svg>';
            break;
        case CHANNEL_DELETED_EVENT:
            svg = '<svg viewBox="-6 -6 32 32" role="presentation" focusable="false" class="app-svg icons-delete"><g class="icons-default-fill"><path class="icons-unfilled" d="M11.5 4C11.5 3.17157 10.8284 2.5 10 2.5C9.17157 2.5 8.5 3.17157 8.5 4H7.5C7.5 2.61929 8.61929 1.5 10 1.5C11.3807 1.5 12.5 2.61929 12.5 4H17C17.2761 4 17.5 4.22386 17.5 4.5C17.5 4.77614 17.2761 5 17 5H16.446L15.1499 16.2292C15.0335 17.2384 14.179 18 13.1631 18H6.83688C5.821 18 4.9665 17.2384 4.85006 16.2292L3.553 5H3C2.75454 5 2.55039 4.82312 2.50806 4.58988L2.5 4.5C2.5 4.22386 2.72386 4 3 4H11.5ZM15.438 5H4.561L5.84347 16.1146C5.90169 16.6192 6.32894 17 6.83688 17H13.1631C13.6711 17 14.0983 16.6192 14.1565 16.1146L15.438 5ZM8.5 7.5C8.74546 7.5 8.94961 7.65477 8.99194 7.85886L9 7.9375V14.0625C9 14.3041 8.77614 14.5 8.5 14.5C8.25454 14.5 8.05039 14.3452 8.00806 14.1411L8 14.0625V7.9375C8 7.69588 8.22386 7.5 8.5 7.5ZM11.5 7.5C11.7455 7.5 11.9496 7.65477 11.9919 7.85886L12 7.9375V14.0625C12 14.3041 11.7761 14.5 11.5 14.5C11.2545 14.5 11.0504 14.3452 11.0081 14.1411L11 14.0625V7.9375C11 7.69588 11.2239 7.5 11.5 7.5Z"></path></g></svg>';
            break;
        case CHANNEL_ADDED_EVENT:
        case CHANNEL_DESC_UPDATED_EVENT:
        case CHANNEL_RENAMED_EVENT:
        case TEAM_DESC_UPDATED_EVENT:
        case TEAM_RENAMED_EVENT:
            svg = '<svg viewBox="-6 -6 32 32" role="presentation" class="app-svg icons-channel-icon" focusable="false"><g class="icons-default-fill"><path d="M3.5 5.5C3.31786 5.5 3.14709 5.5487 3 5.63378C2.7011 5.80669 2.5 6.12986 2.5 6.5C2.5 6.87014 2.7011 7.19331 3 7.36622C3.14709 7.4513 3.31786 7.5 3.5 7.5C3.68214 7.5 3.85291 7.4513 4 7.36622C4.2989 7.19331 4.5 6.87014 4.5 6.5C4.5 6.12986 4.2989 5.80669 4 5.63378C3.85291 5.5487 3.68214 5.5 3.5 5.5Z"></path><path d="M3.5 4.5C3.39744 4.5 3.29668 4.50772 3.19828 4.52261C3.57881 3.6276 4.46611 3 5.5 3H14.5C15.8807 3 17 4.11929 17 5.5V14.5C17 15.8807 15.8807 17 14.5 17H5.5C4.11929 17 3 15.8807 3 14.5V8.43699C3.15981 8.47812 3.32735 8.5 3.5 8.5C3.67265 8.5 3.84019 8.47812 4 8.43699V14.5C4 15.3284 4.67157 16 5.5 16H14.5C15.3284 16 16 15.3284 16 14.5V5.5C16 4.67157 15.3284 4 14.5 4H5.5C4.98679 4 4.53378 4.25774 4.26334 4.65083C4.02812 4.55363 3.77032 4.5 3.5 4.5Z"></path><path d="M7 8.5C7 8.22386 7.22386 8 7.5 8H12.5C12.7761 8 13 8.22386 13 8.5C13 8.77614 12.7761 9 12.5 9H7.5C7.22386 9 7 8.77614 7 8.5Z"></path><path d="M7.5 11C7.22386 11 7 11.2239 7 11.5C7 11.7761 7.22386 12 7.5 12H10.5C10.7761 12 11 11.7761 11 11.5C11 11.2239 10.7761 11 10.5 11H7.5Z"></path></g></svg>';
            break;
        case CHANNEL_SET_AS_FAV_EVENT:
            svg = '<svg viewBox="-6 -6 32 32" role="presentation" class="app-svg icons-show-filled" focusable="false"><g class="icons-default-fill"><path d="M3.25909 11.6021C3.94254 8.32689 6.79437 6 10 6C13.2057 6 16.0574 8.32688 16.7409 11.6021C16.7974 11.8725 17.0622 12.0459 17.3325 11.9895C17.6029 11.933 17.7763 11.6682 17.7199 11.3979C16.9425 7.67312 13.6934 5 10 5C6.3066 5 3.05742 7.67311 2.28017 11.3979C2.22377 11.6682 2.39718 11.933 2.6675 11.9895C2.93782 12.0459 3.20268 11.8725 3.25909 11.6021Z"></path><path d="M9.98953 8C11.9225 8 13.4895 9.567 13.4895 11.5C13.4895 13.433 11.9225 15 9.98953 15C8.05653 15 6.48953 13.433 6.48953 11.5C6.48953 9.567 8.05653 8 9.98953 8Z"></path></g></svg>';
            break;
        case CHANNEL_UNSET_AS_FAV_EVENT:
            svg = '<svg role="presentation" focusable="false" class="app-svg icons-hide-filled" viewBox="-6 -6 32 32"><g class="icons-default-fill"><path d="M2.85355 2.14645C2.65829 1.95118 2.34171 1.95118 2.14645 2.14645C1.95118 2.34171 1.95118 2.65829 2.14645 2.85355L5.64526 6.35237C3.97039 7.49178 2.72334 9.27383 2.28011 11.3979C2.22371 11.6682 2.39712 11.933 2.66744 11.9895C2.93776 12.0459 3.20262 11.8725 3.25903 11.6021C3.66284 9.66698 4.82362 8.06289 6.3671 7.07421L7.94894 8.65604C7.06509 9.29133 6.48947 10.3284 6.48947 11.5C6.48947 13.433 8.05647 15 9.98947 15C11.161 15 12.1981 14.4244 12.8334 13.5405L17.1464 17.8536C17.3417 18.0488 17.6583 18.0488 17.8536 17.8536C18.0488 17.6583 18.0488 17.3417 17.8536 17.1464L2.85355 2.14645Z"></path><path d="M10.1238 8.00253L13.4869 11.3657C13.418 9.5395 11.95 8.07143 10.1238 8.00253Z"></path><path d="M7.53104 5.4098L8.3341 6.21286C8.87141 6.07353 9.43009 6 9.99995 6C13.2056 6 16.0574 8.32688 16.7409 11.6021C16.7973 11.8725 17.0622 12.0459 17.3325 11.9895C17.6028 11.933 17.7762 11.6682 17.7198 11.3979C16.9425 7.67312 13.6934 5 9.99995 5C9.14478 5 8.31342 5.14331 7.53104 5.4098Z"></path></g></svg>';
            break;
        case MEMBERS_ADDED_EVENT:
        case MEMBERS_JOINED_EVENT:
            if (CHANNEL) {
                svg = '<svg viewBox="-6 -6 32 32" role="presentation" class="app-svg icons-add-participant"><g class="icons-default-fill icons-unfilled"><path fill-rule="evenodd" clip-rule="evenodd" d="M9 2C6.79086 2 5 3.79086 5 6C5 8.20914 6.79086 10 9 10C11.2091 10 13 8.20914 13 6C13 3.79086 11.2091 2 9 2ZM6 6C6 4.34315 7.34315 3 9 3C10.6569 3 12 4.34315 12 6C12 7.65685 10.6569 9 9 9C7.34315 9 6 7.65685 6 6Z"></path><path d="M4.00873 11C2.90315 11 2 11.8869 2 13C2 14.6912 2.83281 15.9663 4.13499 16.7966C5.41697 17.614 7.14526 18 9 18C9.41085 18 9.8155 17.9811 10.2105 17.9427C9.97298 17.6472 9.7654 17.3266 9.59233 16.9855C9.39798 16.9951 9.20041 17 9 17C7.26489 17 5.74318 16.636 4.67262 15.9534C3.62226 15.2837 3 14.3088 3 13C3 12.4467 3.44786 12 4.00873 12L9.59971 12C9.7826 11.6422 10.0035 11.3071 10.2572 11L4.00873 11Z"></path><path fill-rule="evenodd" clip-rule="evenodd" d="M14.5 19C16.9853 19 19 16.9853 19 14.5C19 12.0147 16.9853 10 14.5 10C12.0147 10 10 12.0147 10 14.5C10 16.9853 12.0147 19 14.5 19ZM14.5 12C14.7761 12 15 12.2239 15 12.5V14H16.5C16.7761 14 17 14.2239 17 14.5C17 14.7761 16.7761 15 16.5 15H15V16.5C15 16.7761 14.7761 17 14.5 17C14.2239 17 14 16.7761 14 16.5V15H12.5C12.2239 15 12 14.7761 12 14.5C12 14.2239 12.2239 14 12.5 14H14V12.5C14 12.2239 14.2239 12 14.5 12Z"></path></g></svg>';
            } else {
                svg = '<svg role="presentation" focusable="false" viewBox="2 2 16 16" class="cd vd ve vf vg" style="overflow: visible;"><g class="ui-icon__outline cd"><path fill-rule="evenodd" clip-rule="evenodd" d="M9 2C6.79086 2 5 3.79086 5 6C5 8.20914 6.79086 10 9 10C11.2091 10 13 8.20914 13 6C13 3.79086 11.2091 2 9 2ZM6 6C6 4.34315 7.34315 3 9 3C10.6569 3 12 4.34315 12 6C12 7.65685 10.6569 9 9 9C7.34315 9 6 7.65685 6 6Z"></path><path d="M4.00873 11C2.90315 11 2 11.8869 2 13C2 14.6912 2.83281 15.9663 4.13499 16.7966C5.41697 17.614 7.14526 18 9 18C9.41085 18 9.8155 17.9811 10.2105 17.9427C9.97298 17.6472 9.7654 17.3266 9.59233 16.9855C9.39798 16.9951 9.20041 17 9 17C7.26489 17 5.74318 16.636 4.67262 15.9534C3.62226 15.2837 3 14.3088 3 13C3 12.4467 3.44786 12 4.00873 12L9.59971 12C9.7826 11.6422 10.0035 11.3071 10.2572 11L4.00873 11Z"></path><path fill-rule="evenodd" clip-rule="evenodd" d="M14.5 19C16.9853 19 19 16.9853 19 14.5C19 12.0147 16.9853 10 14.5 10C12.0147 10 10 12.0147 10 14.5C10 16.9853 12.0147 19 14.5 19ZM14.5 12C14.7761 12 15 12.2239 15 12.5V14H16.5C16.7761 14 17 14.2239 17 14.5C17 14.7761 16.7761 15 16.5 15H15V16.5C15 16.7761 14.7761 17 14.5 17C14.2239 17 14 16.7761 14 16.5V15H12.5C12.2239 15 12 14.7761 12 14.5C12 14.2239 12.2239 14 12.5 14H14V12.5C14 12.2239 14.2239 12 14.5 12Z"></path></g></svg>';
                iconDiv.className = 'system-msg-smaller-svg';
            }
            break;
        case MEMBERS_DELETED_EVENT:
        case MEMBERS_LEFT_EVENT:
            if (CHANNEL) {
                svg = '<svg viewBox="-6 -6 32 32" role="presentation" class="app-svg icons-leave" focusable="false"><path class="icons-default-fill" d="M6 2C4.89543 2 4 2.89543 4 4V16C4 17.1046 4.89543 18 6 18H10.2572C10.0035 17.6929 9.78261 17.3578 9.59971 17H6C5.44772 17 5 16.5523 5 16V4C5 3.44772 5.44772 3 6 3H14C14.5523 3 15 3.44772 15 4V9.02242C15.3434 9.05337 15.6777 9.11588 16 9.20703V4C16 2.89543 15.1046 2 14 2H6ZM8 10C8 10.5523 7.55228 11 7 11C6.44772 11 6 10.5523 6 10C6 9.44771 6.44772 9 7 9C7.55228 9 8 9.44771 8 10ZM19 14.5C19 16.9853 16.9853 19 14.5 19C12.0147 19 10 16.9853 10 14.5C10 12.0147 12.0147 10 14.5 10C16.9853 10 19 12.0147 19 14.5ZM12.1468 14.146L12.1443 14.1486C12.0974 14.196 12.062 14.2505 12.0379 14.3086C12.0139 14.3667 12.0004 14.4303 12 14.497L12 14.5L12 14.503C12.0004 14.5697 12.0139 14.6333 12.0379 14.6914C12.0623 14.7504 12.0985 14.8056 12.1464 14.8536L14.1464 16.8536C14.3417 17.0488 14.6583 17.0488 14.8536 16.8536C15.0488 16.6583 15.0488 16.3417 14.8536 16.1464L13.7071 15H16.5C16.7761 15 17 14.7761 17 14.5C17 14.2239 16.7761 14 16.5 14H13.7071L14.8536 12.8536C15.0488 12.6583 15.0488 12.3417 14.8536 12.1464C14.6583 11.9512 14.3417 11.9512 14.1464 12.1464L12.1468 14.146Z"></path></svg>';
            } else {
                svg = '<svg role="presentation" focusable="false" viewBox="2 2 16 16" class="fm fn fo fp fq" style="overflow: visible;"><path class="ui-icon__outline fm" d="M9 2C6.79086 2 5 3.79086 5 6C5 8.20914 6.79086 10 9 10C11.2091 10 13 8.20914 13 6C13 3.79086 11.2091 2 9 2ZM6 6C6 4.34315 7.34315 3 9 3C10.6569 3 12 4.34315 12 6C12 7.65685 10.6569 9 9 9C7.34315 9 6 7.65685 6 6ZM4.00873 11C2.90315 11 2 11.8869 2 13C2 14.6912 2.83281 15.9663 4.13499 16.7966C5.41697 17.614 7.14526 18 9 18C9.41085 18 9.8155 17.9811 10.2105 17.9427C9.97298 17.6472 9.7654 17.3266 9.59233 16.9855C9.39798 16.9951 9.20041 17 9 17C7.26489 17 5.74318 16.636 4.67262 15.9534C3.62226 15.2837 3 14.3088 3 13C3 12.4467 3.44786 12 4.00873 12L9.59971 12C9.7826 11.6422 10.0035 11.3071 10.2572 11L4.00873 11ZM19 14.5C19 16.9853 16.9853 19 14.5 19C12.0147 19 10 16.9853 10 14.5C10 12.0147 12.0147 10 14.5 10C16.9853 10 19 12.0147 19 14.5ZM12.1468 14.146L12.1443 14.1486C12.0974 14.196 12.062 14.2505 12.0379 14.3086C12.0139 14.3667 12.0004 14.4303 12 14.497L12 14.5L12 14.503C12.0004 14.5697 12.0139 14.6333 12.0379 14.6914C12.0623 14.7504 12.0985 14.8056 12.1464 14.8536L14.1464 16.8536C14.3417 17.0488 14.6583 17.0488 14.8536 16.8536C15.0488 16.6583 15.0488 16.3417 14.8536 16.1464L13.7071 15H16.5C16.7761 15 17 14.7761 17 14.5C17 14.2239 16.7761 14 16.5 14H13.7071L14.8536 12.8536C15.0488 12.6583 15.0488 12.3417 14.8536 12.1464C14.6583 11.9512 14.3417 11.9512 14.1464 12.1464L12.1468 14.146Z"></path></svg>';
                iconDiv.className = 'system-msg-smaller-svg';
            }
            break;
        case MEMBER_ROLE_UPDATED_EVENT:
            svg = '<svg viewBox="-6 -6 32 32" role="presentation" class="app-svg icons-promote-member"><path class="icons-default-fill" d="M16 8C16 9.777 15.2275 11.3736 14 12.4722V17.5002C14 17.6784 13.9051 17.8432 13.751 17.9327C13.5968 18.0221 13.4067 18.0227 13.2519 17.9343L10 16.0761L6.74807 17.9343C6.59332 18.0227 6.40319 18.0221 6.24904 17.9327C6.09488 17.8432 6 17.6784 6 17.5002V12.4722C4.7725 11.3736 4 9.777 4 8C4 4.68629 6.68629 2 10 2C13.3137 2 16 4.68629 16 8ZM10 14C8.90714 14 7.88252 13.7078 7 13.1973V16.6386L9.75193 15.0661C9.90565 14.9782 10.0944 14.9782 10.2481 15.0661L13 16.6386V13.1973C12.1175 13.7078 11.0929 14 10 14ZM10 13C12.7614 13 15 10.7614 15 8C15 5.23858 12.7614 3 10 3C7.23858 3 5 5.23858 5 8C5 10.7614 7.23858 13 10 13Z"></path></svg>';
            break;
        case TEAMS_APP_INSTALLED_EVENT:
        case TEAMS_APP_REMOVED_EVENT:
            svg = '<svg viewBox="-6 -6 32 32" role="presentation" class="app-svg icons-apps icons-discover-apps"><g class="icons-default-fill"><path class="icons-unfilled" d="M4.5 17C3.7203 17 3.07955 16.4051 3.00687 15.6445L3 15.5V4.5C3 3.7203 3.59489 3.07955 4.35554 3.00687L4.5 3H9C9.7797 3 10.4204 3.59489 10.4931 4.35554L10.5 4.5V4.75443L12.6886 2.48518C13.2276 1.92599 14.0959 1.87568 14.6956 2.34707L14.8118 2.4483L17.5694 5.17295C18.1219 5.71884 18.1614 6.58769 17.68 7.18413L17.5767 7.29961L15.266 9.49943L15.5 9.5C16.2797 9.5 16.9204 10.0949 16.9931 10.8555L17 11V15.5C17 16.2797 16.4051 16.9204 15.6445 16.9931L15.5 17H4.5ZM9.5 10.5H4V15.5C4 15.7148 4.13542 15.8979 4.32553 15.9687L4.41012 15.9919L4.5 16H9.5V10.5ZM15.5 10.5H10.5V16H15.5C15.7455 16 15.9496 15.8231 15.9919 15.5899L16 15.5V11C16 10.7545 15.8231 10.5504 15.5899 10.5081L15.5 10.5ZM10.5 7.70943V9.49943H12.29L10.5 7.70943ZM9 4H4.5C4.25454 4 4.05039 4.17688 4.00806 4.41012L4 4.5V9.5H9.5V4.5C9.5 4.28522 9.36458 4.10207 9.17447 4.03128L9.08988 4.00806L9 4ZM14.1222 3.17265C13.9396 2.99652 13.6692 2.98155 13.4768 3.12004L13.4086 3.17915L10.7926 5.89329C10.6271 6.065 10.6086 6.32501 10.7356 6.51644L10.799 6.59383L13.4147 9.20955C13.5826 9.37746 13.8409 9.40135 14.0345 9.27931L14.1131 9.21806L16.8708 6.59139C17.0433 6.41679 17.061 6.14725 16.9248 5.95319L16.8665 5.88429L14.1222 3.17265Z"></path></g></svg>';
            break;
        default:
            break;
    }
    $(iconDiv).html(svg);
    parentDiv.appendChild(iconDiv);
}

function _setCallEventType(msg, callEventType) {
    if (callEventType === 'call') {
        return msg.replace('{callEventType}', 'Call');
    } else if (callEventType === 'meeting') {
        return msg.replace('{callEventType}', 'Meeting');
    } else {
        // Support screenShare, unknownFutureValue
        return msg.replace('{callEventType}', callEventType);
    }
}

function _getMembersAddedToChatEventMsg(eventDetail) {
    let msg = '{initiator} added {participants} to the chat';
    let initiator = eventDetail.initiator;
    let membersAdded = eventDetail.members.filter(user => user.id !== initiator.user.id);
    msg = msg.replace('{participants}', _concatMembersDisplayNames(membersAdded));

    if (eventDetail.visibleHistoryStartDateTime === SHARED_ALL_HISTORY) {
        msg += ' and shared all chat history';
    }
    msg += '.';
    return msg;
}

function _getMembersDeletedFromChatEventMsg(eventDetail) {
    let initiator = eventDetail.initiator;
    let membersRemoved = eventDetail.members;

    if (membersRemoved.length === 1
            && membersRemoved[0].id === initiator.user.id) {
        return '{initiator} left the chat.';
    }

    let msg = '{initiator} removed {participants} from the chat.';
    return msg;
}

function _getMembersAddedToChannelEventMsg(eventDetail) {
    return _getChannelMembersAddedDeletedMsg(eventDetail, 'added');
}

function _getMembersDeletedFromChannelEventMsg(eventDetail) {
    return _getChannelMembersAddedDeletedMsg(eventDetail, 'removed');
}

function _getChannelMembersAddedDeletedMsg(eventDetail, sAction) {
    let toFromString = sAction === 'added' ? 'to' : 'from';
    let initiator = eventDetail.initiator;
    let membersAdded = _concatMembersDisplayNames(eventDetail.members);
    let msg = '';

    if (channelProperties.membershipType
            && channelProperties.membershipType.toLowerCase() === 'private') {
        msg = '{initiator} has '
            + sAction
            + ' {participants} ' + toFromString + ' the channel.';
    } else {
        if (initiator.user) {
            if (eventDetail.members.length === 1
                    && eventDetail.members[0].id === initiator.user.id) {
                msg = '{participants} '
                    + (sAction === 'added' ? 'joined' : 'left')
                    + ' the team.';
            } else {
                msg = '{initiator} has '
                    + sAction
                    + ' {participants} ' + toFromString + ' the team.';
            }
        } else {
            msg = '{participants} has been '
                + sAction
                + ' ' + toFromString + ' the team.';
        }
    }
    return msg;
}

function _concatMembersDisplayNames(membersArr) {
    let msg = '';
    for (let i = 0; i < membersArr.length; i++) {
        let user;
        if (membersArr[i].participant) {
            user = membersArr[i].participant.user;
        } else {
            user = membersArr[i];
        }
        let displayName = _getDisplayName(user);
        displayName = _removeSpanTag(displayName);
        msg += displayName;
        if (i === membersArr.length - 2) {
            msg += ' and ';
        } else if (i < membersArr.length - 1) {
            msg += ', ';
        }
    }
    return msg;
}

function _getInitiatorDisplayName(initiator) {
    let displayName = '';
    if (initiator) {
        if (initiator.user) {
            displayName = _getDisplayName(initiator.user);
            displayName = _removeSpanTag(displayName);
        } else if (initiator.application) {
            displayName = initiator.application.displayName;
            if (!displayName) {
                displayName = 'Unknown App (' + initiator.application.id + ')'
            }
        }
    }
    return displayName;
}

function _removeSpanTag(htmlStringTag) {
    if (htmlStringTag.endsWith('</span>')) {
        htmlStringTag = htmlStringTag.replace('</span>', '');
        htmlStringTag = htmlStringTag.replace(/<span.*>/ig, '');
    }
    return htmlStringTag;
}

function _formatCallDuration(duration) {
    let period = duration.substring(
        duration.indexOf('P') + 1,
        duration.lastIndexOf('T')
    );
    if (period) {
        period = period.replace('Y', 'Y ')
            .replace('M', 'M ')
            .replace('W', 'W ')
            .replace('D', 'D ');
    } else {
        period = '';
    }
    let time = duration.substring(
        duration.indexOf('T') + 1
    );
    if (time) {
        time = time.replace('H', 'h ')
            .replace('M', 'm ')
            .replace('S', 's');
    } else {
        time = '';
    }
    return period + time;
}

function _getResource(resource) {
    if (isCBS()) {
        return '/cbs/getImages/teams/' + resource + '?type=img';
    }
    if (isExport()) {
        return '../' + resource;
    }
    return resource;
}

/**
 * channel profile json
 * {
 *   "id": "string (identifier)",
 *   "replyToId": "string (identifier)",
 *   "from": {"@odata.type": "microsoft.graph.chatMessageFromIdentitySet"},
 *   "etag": "string",
 *   "messageType": "string",
 *   "createdDateTime": "string (timestamp)",
 *   "lastModifiedDateTime": "string (timestamp)",
 *   "lastEditedDateTime": "string (timestamp)",
 *   "deletedDateTime": "string (timestamp)",
 *   "subject": "string",
 *   "body": {"@odata.type": "microsoft.graph.itemBody"},
 *   "summary": "string",
 *   "attachments": [{"@odata.type": "microsoft.graph.chatMessageAttachment"}],
 *   "mentions": [{"@odata.type": "microsoft.graph.chatMessageMention"}],
 *   "importance": "string",
 *   "reactions": [{"@odata.type": "microsoft.graph.chatMessageReaction"}],
 *   "locale": "string",
 *   "policyViolation": {"@odata.type": "microsoft.graph.chatMessagePolicyViolation"},
 *   "chatId": "string",
 *   "channelIdentity": {"@odata.type": "microsoft.graph.channelIdentity"},
 *   "webUrl": "string"
 * }
 */
function _insertMessage(message) {
    if (message == null) {
        return;
    }
    if (message.messageType !== 'message') {
        _handleSystemEvent(message, null);
        return;
    }

    let teamsBodyDOM = document.getElementById('teams-body');
    let rowDiv = document.createElement('div');
    let msgBaseContainer = document.createElement('div');
    let msgContainer = document.createElement('div');

    let isImportant = message.importance === 'high';
    let isUrgent = message.importance === 'urgent';
    let hasMentions = message.mentions && message.mentions.length != 0;
    let isSelfMessage = _isSelfMessage(message);
    let isDeleted = message.deletedDateTime;

    rowDiv.className = 'd-flex';
    msgBaseContainer.className = message.deletedDateTime ? 'msg-base-container msg-base-container-deleted' : 'msg-base-container';
    msgBaseContainer.id = message.id;
    if (!CHANNEL && isSelfMessage) {
        $(rowDiv).addClass('justify-content-end');
        $(rowDiv).addClass('end-container');
        $(msgContainer).addClass('msg-container-send');

        // Importance icon
        if (isImportant || isUrgent) {
            $(msgContainer).addClass('important-urgent-container');
            _insertImportanceIcon(msgContainer, isImportant, isUrgent, false, message.attachments);
        }
        userDisplayName = '';
    } else {
        if (CHANNEL) {
            $(rowDiv).addClass('channel-msg-container');
        }
        $(rowDiv).addClass('justify-content-start');
        $(rowDiv).addClass('start-container');
        $(msgContainer).addClass('msg-container');

        _insertAvatarIcon(rowDiv, message);

        // Importance and mention icon
        if (CHANNEL) {
            if (isImportant || isUrgent) {
                $(msgContainer).addClass('important-urgent-container');
            } else if (isSelfMessage) {
                $(msgContainer).addClass('self-msg-reply-container');
            }
            _insertImportanceIcon(msgContainer, isImportant, isUrgent, false, message.attachments);
        } else {
            if (isImportant || isUrgent || hasMentions) {
                $(msgContainer).addClass('important-urgent-container');
                _insertImportanceIcon(msgContainer, isImportant, isUrgent, hasMentions, message.attachments);
            }
        }
        if (isDeleted && !message.from) {
            userDisplayName = 'Unknown User';
        } else if (message.from.user) {
            userDisplayName = _getDisplayName(message.from.user);
        } else if (message.from.application) {
            userDisplayName = message.from.application.displayName;
            if (!userDisplayName) {
                userDisplayName = message.from.application.id;
            }
        }
    }

    // Message time, display name, edited time
    let messageTimeLine = _insertMessageTime(
        message.createdDateTime,
        userDisplayName,
        message.lastEditedDateTime,
        message.deletedDateTime,
        isSelfMessage);

    // Reactions
    if (message.reactions && message.reactions.length != 0) {
        _insertReactions(messageTimeLine, message.reactions);
    }
    msgContainer.appendChild(messageTimeLine);

    // Subject
    if (message.subject) {
        _insertSubject(msgContainer, message.subject);
    }

    // Important/Urgent message header
    _insertImportanceTextTag(msgContainer, isImportant, isUrgent);

    // Message body
    msgContainer.appendChild(_insertMessageBody(
        message.body,
        message.mentions,
        message.attachments,
        message.restoreContents,
        CHANNEL || isSelfMessage)
    );

    msgBaseContainer.appendChild(msgContainer);
    rowDiv.appendChild(msgBaseContainer);
    if (!teamsBodyDOM.innerHTML) {
        teamsBodyDOM.appendChild(rowDiv);
    } else {
        teamsBodyDOM.insertBefore(rowDiv, teamsBodyDOM.childNodes[0]);
    }
}

/**
 * Parent messages will be loaded first so that reply messages
 * can be attached to them.
 * Assumption: Parent and corresponding reply messages are on the same batch of response
 */
function _insertChannelReplyMessages(replyList) {
    replyList.forEach(message => {
        _insertChannelReplyMessage(message);
    });
}

function _insertChannelReplyMessage(message) {
    let msgBaseContainer = document.getElementById(message.replyToId);
    if (!msgBaseContainer) {
        // Skip if parent message does not rendered yet
        return;
    }
    if (message.messageType !== 'message') {
        _handleSystemEvent(message, msgBaseContainer);
        return;
    }
    let msgContainer = document.createElement('div');
    msgContainer.className = 'msg-reply-container';
    msgContainer.className = message.deletedDateTime ? 'msg-reply-container msg-reply-container-deleted' : 'msg-reply-container';
    let msgBody = document.createElement('div');

    let isImportant = message.importance === 'high';
    let isUrgent = message.importance === 'urgent';
    let hasMentions = message.mentions && message.mentions.length != 0;
    let isSelfMessage = _isSelfMessage(message);

    _insertAvatarIcon(msgContainer, message);

    // Importance icon
    if (isImportant || isUrgent) {
        $(msgContainer).addClass('important-urgent-container');
    } else if (isSelfMessage) {
        $(msgContainer).addClass('self-msg-reply-container');
    }
    _insertImportanceIcon(msgBody, isImportant, isUrgent, false, message.attachments);

    // Message time, display name, edited time
    let userDisplayName = _getDisplayName(message.from.user);
    if (message.deletedDateTime && !message.from) {
        userDisplayName = 'Unknown User';
    } else if (message.from.user) {
        userDisplayName = _getDisplayName(message.from.user);
    } else if (message.from.application) {
        userDisplayName = message.from.application.displayName;
        if (!userDisplayName) {
            userDisplayName = message.from.application.id;
        }
    }
    let messageTimeLine = _insertMessageTime(
        message.createdDateTime,
        userDisplayName,
        message.lastEditedDateTime,
        isSelfMessage);

    // Reactions
    if (message.reactions && message.reactions.length != 0) {
        _insertReactions(messageTimeLine, message.reactions);
    }
    msgBody.appendChild(messageTimeLine);

    // Subject
    if (message.subject) {
        _insertSubject(msgContainer, message.subject);
    }

    // Important/Urgent message header
    _insertImportanceTextTag(msgBody, isImportant, isUrgent);

    // Message body
    msgBody.appendChild(_insertMessageBody(
        message.body,
        message.mentions,
        message.attachments,
        true)
    );
    msgBody.style.marginLeft = '10px';
    msgContainer.appendChild(msgBody);
    if (!msgBaseContainer.getElementsByClassName('msg-reply-container')) {
        msgBaseContainer.appendChild(msgContainer);
    } else {
        msgBaseContainer.insertBefore(msgContainer, msgBaseContainer.childNodes[1]);
    }
}

function _insertHostedContents(hostedContents) {
    if (!hostedContentsList) {
        hostedContentsList = [];
    }
    hostedContentsList = hostedContents;
}

function _insertAttachments(attachments) {
    if (!attachmentsList) {
        attachmentsList = [];
    }
    attachmentsList = attachments;
}
/**
 * Insert date line splitter
 */
function _insertDateLine(date) {
    let teamsBodyDOM = document.getElementById('teams-body');

    let dateLineDiv = document.createElement('div');
    dateLineDiv.className = 'message-received-date row flex-nowrap';

    let divBefore = document.createElement('div');
    divBefore.className = 'message-received-date-line col';
    dateLineDiv.appendChild(divBefore);

    let span = document.createElement('span');
    span.className = 'col-sm-auto';
    span.innerHTML = _formatDate(date);
    dateLineDiv.appendChild(span);

    let divAfter = document.createElement('div');
    divAfter.className = 'message-received-date-line col';
    dateLineDiv.appendChild(divAfter);

    if (!teamsBodyDOM.innerHTML) {
        teamsBodyDOM.appendChild(dateLineDiv);
    } else {
        teamsBodyDOM.insertBefore(dateLineDiv, teamsBodyDOM.childNodes[0]);
    }
}

/**
 * Insert Message Time
 */
function _insertMessageTime(dateTimeString, displayName, editedTimeString, deletedTimeString, isSelfMessage) {
    let timeDiv = document.createElement('div');
    timeDiv.className = 'sender-time-date';

    // Display name
    if (displayName) {
        timeDiv.innerHTML = ' ' + displayName + ' ';
    }

    // Date created
    let createdDate = new Date(dateTimeString);
    let createdSpan = document.createElement('span');
    createdSpan.innerHTML = _formatDateTime(createdDate);
    timeDiv.appendChild(createdSpan);

    // Edited time tooltip
    if (deletedTimeString) {
        let deletedDate = new Date(deletedTimeString);
        let deletedSpan = document.createElement('span');
        let deletedTooltip = document.createElement('span');

        deletedSpan.className = 'tooltip';
        deletedTooltip.className = 'tooltiptext';
        if (isSelfMessage) {
            deletedSpan.style.marginLeft = '8px';
        }

        deletedSpan.innerHTML = LANG_DELETED;
        deletedTooltip.innerHTML = LANG_DELETED + ' ' + _formatDateTime(deletedDate);

        deletedSpan.appendChild(deletedTooltip);
        timeDiv.appendChild(deletedSpan);
    } else if (editedTimeString) {
        let editedDate = new Date(editedTimeString);
        let editedSpan = document.createElement('span');
        let editedTooltip = document.createElement('span');

        editedSpan.className = 'tooltip';
        editedTooltip.className = 'tooltiptext';
        if (isSelfMessage) {
            editedSpan.style.marginLeft = '8px';
        }

        editedSpan.innerHTML = LANG_EDITED;
        editedTooltip.innerHTML = LANG_EDITED + ' ' + _formatDateTime(editedDate);

        editedSpan.appendChild(editedTooltip);
        timeDiv.appendChild(editedSpan);
    }
    return timeDiv;
}

/**
 * Insert Reactions
 */
function _insertReactions(parentDiv, reactions) {
    let reactionMap = {
        'like': new Array(),
        'angry': new Array(),
        'sad': new Array(),
        'laugh': new Array(),
        'heart': new Array(),
        'surprised': new Array()
    };

    reactions.forEach(item => {
        reactionMap[item.reactionType].push(item.user.user);
    });


    let reactionsDiv = document.createElement('div');
    reactionsDiv.className = 'reactions';

    // Like
    _insertEachReaction(reactionsDiv, LANG_LIKE, 'lib/images/reactions/like.png', reactionMap['like']);
    // Heart
    _insertEachReaction(reactionsDiv, LANG_HEART, 'lib/images/reactions/heart.png', reactionMap['heart']);
    // Laugh
    _insertEachReaction(reactionsDiv, LANG_LAUGH, 'lib/images/reactions/laugh.png', reactionMap['laugh']);
    // Surprised
    _insertEachReaction(reactionsDiv, LANG_SURPRISED, 'lib/images/reactions/surprised.png', reactionMap['surprised']);
    // Sad
    _insertEachReaction(reactionsDiv, LANG_SAD, 'lib/images/reactions/sad.png', reactionMap['sad']);
    // Angry
    _insertEachReaction(reactionsDiv, LANG_ANGRY, 'lib/images/reactions/angry.png', reactionMap['angry']);

    parentDiv.appendChild(reactionsDiv);
}

function _insertEachReaction(reactionsDiv, reactType, iconSource, users) {
    if (users.length <= 0) {
        return;
    }
    iconSource = _getResource(iconSource);

    let reactDiv = document.createElement('div');
    let icon = document.createElement('img');
    let countDiv = document.createElement('div');
    let reactTooltip = document.createElement('div');

    reactDiv.className = 'tooltip';
    reactTooltip.className = 'tooltiptext';
    icon.className = 'reaction-icon';
    countDiv.className = 'reaction-count';

    icon.src = iconSource;
    countDiv.innerHTML = users.length;
    reactDiv.appendChild(icon);
    reactDiv.appendChild(countDiv);

    // Tooltip contents
    let reactionLine = document.createElement('div');
    let reactionText = document.createElement('div');
    reactionText.innerHTML = ' ' + reactType + ' ' + users.length + ' ' + (users.length > 1 ? LANG_REACTIONS : LANG_REACTION);
    reactionLine.innerHTML = '<img class="reaction-icon" src="' + iconSource +'" />';
    reactionLine.appendChild(reactionText);
    reactTooltip.appendChild(reactionLine);

    let usersDiv = document.createElement('div');
    usersDiv.style.marginLeft = '8px';
    users.forEach(item => {
        let eachUserDiv = document.createElement('div');
        eachUserDiv.className = 'reaction-user-box';
        let iconImg = document.createElement('img');
        let userName = document.createElement('div');
        userName.className = 'reaction-user-name'

        var displayName = _getDisplayName(item);
        if (displayName == null && members != null) {
            for (i = 0; i < members.length; i++) {
                let member = members[i];
                if (member.userId == item.id) {
                    displayName = member.displayName;
                    break;
                }
            }
        }
        _insertUserNameIcon(eachUserDiv, displayName);

        userName.innerHTML = displayName;
        eachUserDiv.appendChild(userName);

        usersDiv.appendChild(eachUserDiv);
    });
    reactTooltip.appendChild(usersDiv);
    reactDiv.appendChild(reactTooltip);
    reactionsDiv.appendChild(reactDiv);
}

/**
 * Insert Message Body
 */
function _insertMessageBody(body, mentions, attachments, restoreContents, isSelfMessage) {
    let msgDiv = document.createElement('div');
    msgDiv.className = 'msg-content';

    // Replace hosted content in the body first
    let bodyContent = _replaceHostedContents(body.content);
    if (body.contentType === 'html') {
        _insertHtmlMessage(
            msgDiv,
            bodyContent,
            attachments,
            mentions,
            isSelfMessage);
    } else {
        _insertPlainTextMessage(msgDiv, bodyContent, attachments);
    }
    return msgDiv;
}

/**
 * Insert plain text message
 */
function _insertPlainTextMessage(msgDiv, content, attachments) {
    msgDiv.innerHTML += content;
    if (attachments && attachments.length != 0) {
            _checkAttachments(msgDiv, attachments);
    }
}

/**
 * Insert HTML message
 */
function _insertHtmlMessage(msgDiv, content, attachments, mentions, isSelfMessage) {
    msgDiv.innerHTML += content;

    _checkHtmlLinks(msgDiv);

    if (attachments && attachments.length != 0) {
        _checkAttachments(msgDiv, attachments);
    }
    if (mentions && mentions.length != 0) {
        _checkMentions(msgDiv, mentions, isSelfMessage);
    }
}

/**
 * Check links from message
 */
function _checkHtmlLinks(msgDiv) {
    let aQueries = msgDiv.querySelectorAll('a');
    aQueries.forEach(function (aQuery) {
        let href = aQuery.getAttribute('href');
        if (href !== null) {
            aQuery.removeAttribute('target');
            _checkHRef(msgDiv, href);
        }
    });
}

/**
 * Check mentions
 */
function _checkMentions(msgDiv, mentions, isSelfMessage) {
    mentions.forEach(item => {
        let tempHTML = msgDiv.innerHTML;
        let findString = new RegExp('<at id="' + item.id + '">.*?</at>');
        let userMentioned = false;
        if (!CHANNEL && item.mentioned.user !== null) {
            userMentioned = item.mentioned.user.id === me.id;
        }
        let className = userMentioned ? (isSelfMessage ? 'mention-text-send' : 'mention-text') : 'mention-text-send';
        let replaceString = '<span id="' + item.id + '" class="' + className + '">' + item.mentionText + ' </span>';

        msgDiv.innerHTML = tempHTML.replace(findString, replaceString);
        findString = new RegExp('<at id=\\"' + item.id + '\\">.*?</at>');
        msgDiv.innerHTML = tempHTML.replace(findString, replaceString);
    });
}

/**
 * Insert attachment elements
 */
function _checkAttachments(parentDiv, attachments) {
    // Check if the HTML message contains attachment and apply the elements
    var messageHTML = parentDiv.innerHTML;
    attachments.forEach(attachment => {
        let findString = '<attachment id="' + attachment.id + '"></attachment>';
        let idxFindString = messageHTML ? messageHTML.indexOf(findString) : -1;
        let replaceString = _checkLocalAttachment(attachment);
        if (replaceString) {
            replaceString = _replaceHostedContents(replaceString);
            if (idxFindString > 0) {
                replaceString = "<br>" + replaceString;
            }
            messageHTML = messageHTML.replace(findString, replaceString);
        }
    });
    parentDiv.innerHTML = messageHTML;
}

function _checkLocalAttachment(attachment) {
    if (attachment.contentType === 'application/vnd.microsoft.card.fluid') {
        return _renderLoopComponent(attachment);
    }
    let localAttachment;
    for (let i = 0; attachmentsList && i < attachmentsList.length; i++) {
        if (attachment.id === attachmentsList[i].id) {
            localAttachment = attachmentsList[i];
            break;
        }
    }
    var localUrl = null
    var localSourcePath = null;
    var downloadName = attachment.name;
    if (localAttachment) {
        downloadName = localAttachment.fileName;
        if (localAttachment.localPath) {
            localUrl = localAttachment.localPath;
        }
        if (localAttachment.sourcePath) {
            localSourcePath = localAttachment.sourcePath;
        }
        attachment.contentType = localAttachment.contentType;
    } else {
        // No localAttachment found, try to replace the contentUrl or content
        // Attachment content and contentUrl are mutually exclusive
        if (attachment.contentUrl) {
            let contentUrl = _replaceHostedContents(attachment.contentUrl);
            // Set as localUrl if the url has been replaced
            if (contentUrl != attachment.contentUrl) {
                localUrl = contentUrl;
                attachment.contentUrl = contentUrl;
            }
        }
    }
    var replaceString = null;
    if (attachment.contentType === 'reference') {
        // Replace attachment tag with anchor tag
        if (localUrl) {
            replaceString = '<a id="' + attachment.id + '" href="javascript:doLocalDownload(\'' + localUrl + '\')">'
                + attachment.name + '</a>';
        } else if (localSourcePath) {
            replaceString = '<a id="' + attachment.id + '" href="javascript:doQuickDownload(\'' + localSourcePath + '\')">'
                + attachment.name + '</a>';
        } else {
            replaceString = '<a id="' + attachment.id + '" href="' + attachment.contentUrl
                + '" target="_blank">' + attachment.name +'</a>';
        }
    } else if (attachment.contentType === 'messageReference') {
        // Chat reply
        replaceString = _renderChatReply(attachment).innerHTML;
    } else if (attachment.contentType === 'application/vnd.microsoft.card.codesnippet') {
        // Create download link for code snippets
        var contentJson;
        try {
            contentJson = JSON.parse(attachment.content);
        } catch (error) {
            console.log(error);
        }
        let subject = '';
        if (contentJson.name) {
            subject = contentJson.name + ' ';
        }
        subject = '<b>' + subject + '</b>(' + contentJson.language + ')';
        if (localUrl) {
            replaceString = 'Code snippet: <a id="' + attachment.id + '" href="javascript:doLocalDownload(\'' + localUrl + '\')">'
                + subject + '</a>';
        } else {
            if (contentJson.codeSnippetUrl) {
                replaceString = 'Code snippet: <a id="' + attachment.id + '" href="' + contentJson.codeSnippetUrl
                    + '" target="_blank">' + subject + '</a>';
            } else if (localSourcePath) {
                replaceString = 'Code snippet: <a id="' + attachment.id + '" href="javascript:doQuickDownload(\'' + localSourcePath + '\')">' + subject + '</a>';
            }
        }
    } else if (attachment.contentType.startsWith('text')) {
        // Create download link for text
        if (localUrl) {
            replaceString = _renderLocalDownloadLink(attachment.id, localUrl, attachment.name, attachment.name).innerHTML;
        } else if (localSourcePath) {
            replaceString = _renderQuickDownloadLink(attachment.id, localSourcePath, attachment.name, attachment.name).innerHTML;
        } else {
            replaceString = '<a id="' + attachment.id + '" href="' + attachment.contentUrl
                    + '" target="_blank">' + attachment.name +'</a>';
        }
    } else if (attachment.contentType.startsWith('image/')) {
        if (localUrl) {
            replaceString = _renderLocalDownloadLink(attachment.id, localUrl, downloadName, '<img src="' + localUrl + '" class="chat-img-size"/>').innerHTML;
        } else if (localSourcePath) {
            if (isCBS()) {
                replaceString = _renderQuickDownloadLink(
                    attachment.id,
                    localSourcePath,
                    downloadName,
                    '<img src="' + localSourcePath + '" class="chat-img-size"/>'
                ).innerHTML;
            } else {
                replaceString = _renderPreviewMedia(attachment.id, localSourcePath, downloadName, downloadName ).innerHTML;
            }
        } else {
            replaceString = '<a id="' + attachment.id + '" href="' + attachment.contentUrl + '" target="_blank">'
                + '<img src="' + attachment.contentUrl + '" class="chat-img-size"/>' + '</a>';
        }
    } else if (attachment.contentType.startsWith('audio/') ||
               attachment.contentType === 'application/vnd.microsoft.card.audio') {
        // Audio Card
        if (localUrl) {
            replaceString = _renderSingleMediaAttachmentLink(attachment, localUrl, downloadName, 'audio').innerHTML;
        } else if (localSourcePath) {
            if (isCBS()) {
                replaceString = _renderSingleMediaAttachmentLink(attachment, localSourcePath, downloadName, 'audio').innerHTML;
            } else {
                replaceString = _renderPreviewMedia(attachment.id, localSourcePath, downloadName, downloadName).innerHTML;
            }
        } else {
            replaceString = _renderMultipleMediaAttachmentLinks(attachment, 'audio').innerHTML;
        }
    } else if (attachment.contentType.startsWith('video/') ||
               attachment.contentType === 'application/vnd.microsoft.card.video') {
        // Video Card
        if (localUrl) {
            replaceString = _renderSingleMediaAttachmentLink(attachment, localUrl, downloadName, 'video').innerHTML;
        } else if (localSourcePath) {
            if (isCBS()) {
                replaceString = _renderSingleMediaAttachmentLink(attachment, localSourcePath, downloadName, 'video').innerHTML;
            } else {
                replaceString = _renderPreviewMedia(attachment.id, localSourcePath, downloadName, downloadName).innerHTML;
            }
        } else {
            replaceString = _renderMultipleMediaAttachmentLinks(attachment, 'video').innerHTML;
        }
    } else if (attachment.contentType === 'application/vnd.microsoft.card.adaptive') {
        // Adaptive Card
        replaceString = _renderAdaptiveCard(attachment).innerHTML;
    } else if (attachment.contentType === 'application/vnd.microsoft.card.animation') {
        // Animation Card
        // TODO: Further render
        replaceString = '<div>' + attachment.content + '</div>';
    } else if (attachment.contentType === 'application/vnd.microsoft.card.hero') {
        // Hero Card
        replaceString = _renderHeroCard(attachment).innerHTML;
    } else if (attachment.contentType === 'application/vnd.microsoft.card.receipt') {
        // TODO: Further render
        replaceString = '<div>' + attachment.content + '</div>';
    } else if (attachment.contentType === 'application/vnd.microsoft.card.signin') {
        // TODO: Further render
        replaceString = '<div>' + attachment.content + '</div>';
    } else if (attachment.contentType === 'application/vnd.microsoft.card.thumbnail') {
        // Thumbnail Card
        replaceString = _renderThumbnailCard(attachment).innerHTML;
    } else if (attachment.contentType === 'application/vnd.microsoft.teams.messaging-announcementBanner') {
        // Announcement banner
        replaceString = _renderAnnouncement(attachment).innerHTML;
    } else {
        // Unsupported types: content displayed as raw
        if (localUrl) {
            replaceString = _renderLocalDownloadLink(attachment.id, localUrl, attachment.name, attachment.name).innerHTML;
        } else if (localSourcePath) {
            if (isCBS()) {
                replaceString = '<a id="' + attachment.id + '" href="' + localSourcePath + '" download>'
                        + attachment.name + '</a>';
            } else {
                replaceString = '<a id="' + attachment.id + '" href="javascript:doQuickDownload(\'' + localSourcePath + '\')">'
                        + attachment.name + '</a>';
            }
        } else {
            replaceString = '<div>' + attachment.content + '</div>';
        }
    }
    return replaceString;
}

/**
 * Check if attachment is saved locally
 */
/*
function _checkLocalAttachment(contentHtml, attachment) {
    if (!attachmentsList || attachmentsList.length == 0) {
        return null;
    }
    let restoreContent;
    for (let i = 0; i < attachmentsList.length; i++) {
        if (attachment.id === attachmentsList[i].attachmentid) {
            restoreContent = attachmentsList[i];
            break;
        }
    }
    var contentUrl = attachment.contentUrl;
    var content = attachment.content;
    var contentSourcePath = null;
    if (restoreContent) {
        if (restoreContent.localPath) {
            contentUrl = restoreContent.localPath;
        }
        contentSourcePath = restoreContent.sourcePath;
    } else {
        // No restoreContent found, try to replace the contentUrl
        contentUrl = _replaceHostedContents(contentUrl);
        content = _replaceHostedContents(content);
    }
    // Check MIME type
    let findString = '<attachment id="' + attachment.id + '"></attachment>';
    let replaceString = null;
    // Show the content if only the url exist
    if (attachment.contentType.startsWith('image/')) {
        if (contentUrl) {
            if (contentSourcePath) {
                replaceString = '<a id="' + attachment.id + '" href="javascript:doQuickDownload(\'' + contentSourcePath + '\')">'
                    + '<img src="' + contentUrl + '" class="chat-img-size"/>' + '</a>';
            } else {
                replaceString = '<a id="' + attachment.id + '" href="' + contentUrl + '" target="_blank">'
                    + '<img src="' + contentUrl + '" class="chat-img-size"/>' + '</a>';
            }
        }
    } else if (attachment.contentType.startsWith('audio/')) {
        if (contentUrl) {
            replaceString = '<audio controls="" class="chat-audio-size">'
                + '<source src="' + contentUrl + '" type="' + attachment.contentType + '"/></audio>';
        }
    } else if (attachment.contentType.startsWith('video/')) {
        if (contentUrl) {
            replaceString = '<video controls="" class="chat-video-size">'
                + '<source src="' + contentUrl + '" type="' + attachment.contentType + '"/></video>';
        }
    } else if (attachment.contentType === 'application/vnd.microsoft.card.codesnippet') {
        // Create download link for code snippets
        let contentJson = JSON.parse(content);
        let subject = '';
        if (contentJson.name) {
            subject = contentJson.name + ' ';
        }
        subject = '<b>' + subject + '</b>(' + contentJson.language + ')';
        if (contentUrl) {
            replaceString = 'Code snippet: <a id="' + attachment.id + '" href="' + contentUrl
                    + '" target="_blank">' + subject + '</a>';
        } else {
            if (contentJson.codeSnippetUrl) {
                replaceString = 'Code snippet: <a id="' + attachment.id + '" href="' + contentJson.codeSnippetUrl
                    + '" target="_blank">' + subject + '</a>';
            } else if (contentSourcePath) {
                replaceString = 'Code snippet: <a id="' + attachment.id + '" href="javascript:doQuickDownload(\'' + contentSourcePath + '\')">' + subject + '</a>';
            }
        }
    } else if (attachment.contentType.startsWith('text/')) {
        // Create download link for text
        if (contentSourcePath) {
            replaceString = '<a id="' + attachment.id + '" href="javascript:doQuickDownload(\'' + contentSourcePath + '\')">' + attachment.name +'</a>';
        } else {
            replaceString = '<a id="' + attachment.id + '" href="' + contentUrl
                    + '" target="_blank">' + attachment.name +'</a>';
        }
    } else if (attachment.contentType === 'reference') {
                // Replace attachment tag with anchor tag
                replaceString = '<a id="' + attachment.id + '" href="' + contentUrl
                    + '" target="_blank">' + attachment.name +'</a>';
    } else {
        return null;
    }
    if (!replaceString) {
        if (contentSourcePath) {
            replaceString = '<a id="' + attachment.id + '" href="javascript:doQuickDownload(\'' + contentSourcePath + '\')">' + attachment.name +'</a>';
        } else {
            return null;
        }
    }
    return contentHtml.replace(findString, replaceString);
}*/

function _getDisplayName(user) {
    if (user == null) {
        return '';
    }
    let id = user.id;
    let displayName = user.displayName;
    if (displayName) {
        if (id) {
            DISPLAY_NAME_MAP[id] = displayName;
            $('span[name="' + id + '"]').html(displayName);
        }
        return displayName;
    }
    if (id && members) {
        for (i = 0; i < members.length; i++) {
            let member = members[i];
            if (member.userId != id) {
                continue;
            }
            displayName = member.displayName;
            DISPLAY_NAME_MAP[id] = displayName;
            $('span[name="' + id + '"]').html(displayName);
        }
    }
    displayName = DISPLAY_NAME_MAP[id];
    return displayName ? displayName : '<span name=\'' + id + '\'>Unknown User (' + id + ')</span>';
}

function _renderChatReply(attachment) {
    var parentMsgContent;
    try {
        parentMsgContent = JSON.parse(attachment.content);
    } catch (error) {
        console.log(error);
    }
    let parentMsg = document.createElement('div');
    parentMsg.className = 'chat-reply-parent';
    parentMsg.id = attachment.id;

    let sender = document.createElement('div');
    sender.className = 'chat-reply-parent-sender';
    var messageSender = '';
    try {
        messageSender = parentMsgContent.messageSender;
        if (messageSender.user) {
            messageSender = _getDisplayName(messageSender.user);
        }
    } catch (error) {
        console.log(error);
    }
    sender.innerHTML = messageSender
        ? messageSender
        : '';
    parentMsg.appendChild(sender);
    parentMsg.innerHTML += parentMsgContent.messagePreview;

    let tempDiv = document.createElement('div');
    tempDiv.appendChild(parentMsg);
    return tempDiv;
}

function _createMediaElement(id, type, urls) {
    let mediaDiv = document.createElement(type);
    mediaDiv.id = id;
    mediaDiv.className = 'chat-' + type + '-size';
    mediaDiv.controls = true;
    mediaDiv.loop = true;
    mediaDiv.autoplay = false;
    urls.forEach(url => {
        let source = document.createElement('source');
        source.src = url;
        mediaDiv.appendChild(source);
    });
    return mediaDiv;
}

function _renderQuickDownloadLink(id, sourcePath, name, content) {
    let source = document.createElement('a');
    let linkText = document.createTextNode(content);
    source.innerHTML = content;
    source.id = id;
    source.title = name;
    if (isCBS()) {
        source.href = sourcePath;
        source.target = '_blank';
    } else {
        source.href = 'javascript:doQuickDownload(\'' + sourcePath + '\')';
    }
    let tempDiv = document.createElement('div');
    tempDiv.appendChild(source);
    return tempDiv;
}

function _renderPreviewMedia(id, sourcePath, name, content) {
    let source = document.createElement('a');
    let linkText = document.createTextNode(content);
    source.innerHTML = content;
    source.id = id;
    source.title = name;
    source.href = 'javascript:doPreviewMedia(\'' + sourcePath + '\')';
    let tempDiv = document.createElement('div');
    tempDiv.appendChild(source);
    return tempDiv;
}

function _renderLocalDownloadLink(id, sourcePath, name, content) {
    let source = document.createElement('a');
    source.innerHTML = content;
    source.id = id;
    source.title = name;
    if (isCBS()) {
        source.href = sourcePath;
        source.target = '_blank';
    } else {
        source.href = 'javascript:doLocalDownload(\'' + sourcePath + '\')';
    }
    let tempDiv = document.createElement('div');
    tempDiv.appendChild(source);
    return tempDiv;
}

function _renderSingleMediaAttachmentLink(attachment, localPath, fileName, type) {
    let mediaDiv;
    var localMediaUrls = [localPath];
    mediaDiv = _createMediaElement(attachment.id, type, localMediaUrls);
    let tempDiv = document.createElement('div');
    tempDiv.appendChild(mediaDiv);
    return tempDiv;
}

function _renderMultipleMediaAttachmentLinks(attachment, type) {
    var jsonContent;
    try {
        jsonContent = JSON.parse(attachment.content);
    } catch (error) {
        console.log(error);
    }
    let mediaDiv;
    if (!jsonContent) {
        mediaDiv = document.createElement('div');
        mediaDiv.innerHTML = attachment.name;
    } else {
        // Map the local paths first
        var allLocal = true;
        var localMediaUrls = [];
        for (let media of jsonContent.media) {
            let mediaUrl = _replaceHostedContents(media.url);
            if (mediaUrl == media.url) {
                allLocal = false;
            }
            localMediaUrls.push(mediaUrl);
        }
        if (allLocal) {
            mediaDiv = _createMediaElement(attachment.id, type, localMediaUrls);
        } else {
            // We won't use local path preview if any one of the media is not in local.
            // Create download link instead
            mediaDiv = document.createElement('div');
            mediaDiv.innerHTML = attachment.name;
            for (let mediaUrl of localMediaUrls) {
                let source = _renderLocalDownloadLink(attachment.id, mediaUrl, attachment.name, mediaUrl);
                mediaDiv.appendChild(source);
            }
        }
    }
    let tempDiv = document.createElement('div');
    tempDiv.appendChild(mediaDiv);
    return tempDiv;
}

/*
function _renderAudioCard(attachment) {
    let jsonContent = JSON.parse(attachment.content);
    let audio;
    if (!jsonContent) {
        audio = document.createElement('div');
        audio.innerHTML = attachment.name;
    } else {
        // Map the local paths first
        var allLocal = true;
        var localMediaUrls = [];
        for (let media of jsonContent.media) {
            let mediaUrl = _replaceHostedContents(media.url);
            if (mediaUrl == media.url) {
                allLocal = false;
            }
            localMediaUrls.push(mediaUrl);
        }
        if (allLocal) {
            audio = document.createElement('audio');
            audio.id = attachment.id;
            audio.className = 'chat-audio-size';
            audio.controls = true;
            if (jsonContent.autoloop == null || jsonContent.autoloop == true) {
                audio.loop = true;
            }
            if (jsonContent.autostart == null || jsonContent.autostart == true) {
                audio.autoplay = true;
            }
            localMediaUrls.forEach(url => {
                let source = document.createElement('source');
                source.src = url;
                audio.appendChild(source);
            });
        } else {
            // We won't use local path preview if any one of the media is not in local.
            // Create download link instead
            audio = document.createElement('div');
            audio.innerHTML = attachment.name;
            for (let url of localMediaUrls) {
                let source = document.createElement('a');
                let linkText = document.createTextNode(mediaUrl);
                source.appendChild(linkText);
                source.title = mediaUrl;
                source.href = mediaUrl;
                audio.appendChild(source);
            }
        }
    }
    let tempDiv = document.createElement('div');
    tempDiv.appendChild(audio);
    return tempDiv;
}

function _renderVideoCard(attachment) {
    let jsonContent = JSON.parse(attachment.content);
    let video;
    if (!jsonContent) {
        video = document.createElement('div');
        video.innerHTML = attachment.name;
    } else {
        video = document.createElement('video');
        video.id = attachment.id;
        video.className = 'chat-video-size';
        video.controls = true;
        if (jsonContent.autoloop == null || jsonContent.autoloop == true) {
            video.loop = true;
        }
        if (jsonContent.autostart == null || jsonContent.autostart == true) {
            video.autoplay = true;
        }
        jsonContent.media.forEach(media => {
            let source = document.createElement('source');
            source.src = media.url;
            video.appendChild(source);
        });
    }
    let tempDiv = document.createElement('div');
    tempDiv.appendChild(video);
    return tempDiv;
}
*/

function _renderAdaptiveCard(attachment) {
    let adaptiveCard = new AdaptiveCards.AdaptiveCard();
    adaptiveCard.hostConfig = new AdaptiveCards.HostConfig({
        fontFamily: 'Segoe UI, Helvetica Neue, sans-serif'
    });
    adaptiveCard.onExecuteAction = function(action) {
        alert('Action is not supported in restored data');
    }
    try {
        adaptiveCard.parse(JSON.parse(attachment.content));
    } catch (error) {
        console.log(error);
    }
    let renderedCard = adaptiveCard.render();
    renderedCard.id = attachment.id;

    let tempDiv = document.createElement('div');
    tempDiv.appendChild(renderedCard);
    return tempDiv;
}

function _renderAnimationCard() {

}

function _renderHeroCard(attachment) {
    var jsonContent;
    try {
        jsonContent = JSON.parse(attachment.content);
    } catch (error) {
        console.log(error);
    }
    let heroCard = document.createElement('div');
    heroCard.id = attachment.id;
    heroCard.className = 'herocard richcard';

    // Title
    if (jsonContent.title) {
        let title = document.createElement('div');
        title.innerHTML = jsonContent.title;
        title.className = 'card-title';
        heroCard.appendChild(title);
    }

    // Subtitle
    if (jsonContent.subtitle) {
        let subtitle = document.createElement('div');
        subtitle.innerHTML = jsonContent.subtitle;
        subtitle.className = 'card-subtitle';
        heroCard.appendChild(subtitle);
    }

    // Image
    if (jsonContent.images && jsonContent.images.length > 0) {
        let imageElement = document.createElement('div');
        imageElement.style.backgroundImage = 'url("' + jsonContent.images[0].url + '")';
        imageElement.className = 'hero-card-image';
        if (jsonContent.images[0].alt) {
            imageElement.alt = jsonContent.images[0].alt;
            imageElement.title = jsonContent.images[0].alt;
        }
        heroCard.appendChild(imageElement);
    }

    // Text
    if (jsonContent.text) {
        let otherText = document.createElement('div');
        otherText.innerHTML = jsonContent.text;
        otherText.className = 'card-text';
        heroCard.appendChild(otherText);
    }

    // Buttons
    if (jsonContent.buttons && jsonContent.buttons.length > 0) {
        let buttonContainer = document.createElement('div');
        buttonContainer.className = 'button-container';
        jsonContent.buttons.forEach(buttonItem => {
            let buttonElement = document.createElement('button');
            buttonElement.textContent = buttonItem.title;
            buttonContainer.appendChild(buttonElement);
        });
        heroCard.appendChild(buttonContainer);
    }

    let tempDiv = document.createElement('div');
    tempDiv.appendChild(heroCard);
    return tempDiv;
}

function _renderSigninCard() {
    // TODO
}

function _renderReceiptCard(attachment) {
    // TODO
    var jsonContent;
    try {
        jsonContent = JSON.parse(attachment.content);
    } catch (error) {
        console.log(error);
    }
    let receiptCard = document.createElement('div');
    receiptCard.id = attachment.id;
    receiptCard.className = 'richcard receipt-card';

    // Title
    if (jsonContent.title) {
        let title = document.createElement('div');
        title.innerHTML = jsonContent.title;
        title.className = 'card-title';
        receiptCard.appendChild(title);
    }

    // Items
    if (jsonContent.items && jsonContent.items.length > 0) {

    }

    // Facts
    if (jsonContent.facts && jsonContent.facts.length > 0) {

    }

    // Tax

    // Vat

    // Total

    // Buttons
    if (jsonContent.buttons && jsonContent.buttons.length > 0) {
        let buttonContainer = document.createElement('div');
        buttonContainer.className = 'button-container';
        jsonContent.buttons.forEach(buttonItem => {
            let buttonElement = document.createElement('button');
            buttonElement.textContent = buttonItem.title;
            buttonContainer.appendChild(buttonElement);
        });
        receiptCard.appendChild(buttonContainer);
    }

    let tempDiv = document.createElement('div');
    tempDiv.appendChild(receiptCard);
    return tempDiv;
}

function _renderThumbnailCard(attachment) {
    var jsonContent;
    try {
        jsonContent = JSON.parse(attachment.content);
    } catch (error) {
        console.log(error);
    }
    let thumbCard = document.createElement('div');
    thumbCard.id = attachment.id;
    thumbCard.className = 'richcard thumbnail-card';

    // Images
    if (jsonContent.images && jsonContent.images.length > 0) {
        let imageElement = document.createElement('img');
        imageElement.src = jsonContent.images[0].url;
        imageElement.className = 'thumbnail-card-image';
        if (jsonContent.images[0].alt) {
            imageElement.alt = jsonContent.images[0].alt;
            imageElement.title = jsonContent.images[0].alt;
        }
        thumbCard.appendChild(imageElement);
    }

    // Description
    let description = document.createElement('div');

    if (jsonContent.title) {
        let title = document.createElement('div');
        title.innerHTML = jsonContent.title;
        title.className = 'card-title';
        description.appendChild(title);
    }

    // Subtitle
    if (jsonContent.subtitle) {
        let subtitle = document.createElement('div');
        subtitle.innerHTML = jsonContent.subtitle;
        subtitle.className = 'card-subtitle';
        description.appendChild(subtitle);
    }

    // Text
    if (jsonContent.text) {
        let otherText = document.createElement('div');
        otherText.innerHTML = jsonContent.text;
        otherText.className = 'card-text';
        description.appendChild(otherText);
    }

    // Button
    if (jsonContent.buttons && jsonContent.buttons.length > 0) {
        let buttonContainer = document.createElement('div');
        buttonContainer.className = 'button-container';
        jsonContent.buttons.forEach(buttonItem => {
            let buttonElement = document.createElement('button');
            buttonElement.textContent = buttonItem.title;
            buttonContainer.appendChild(buttonElement);
        });
        description.appendChild(buttonContainer);
    }
    thumbCard.appendChild(description);

    // Tap
    if (jsonContent.tap) {
        let altText = jsonContent.tap.title;
        if (jsonContent.tap.value) {
            altText += '\n' + jsonContent.tap.value;
        }
        thumbCard.alt = altText;
        thumbCard.title = altText;
    }
    let tempDiv = document.createElement('div');
    tempDiv.appendChild(thumbCard);
    return tempDiv;
}

function _renderAnnouncement(attachment) {
    var jsonContent;
    try {
        jsonContent = JSON.parse(attachment.content);
    } catch (error) {
        console.log(error);
    }
    let card = document.createElement('div');
    card.id = attachment.id;
    card.className = 'richcard announcement-card';

    if (jsonContent.cardImageType === 'colorTheme') {
        switch (jsonContent.cardImageDetails.colorTheme) {
            case 'periwinkleBlue':
                $(card).addClass('periwrinkle-blue-theme');
                break;
            default:
                // TODO: Support other color themes
                break;
        }
    } else {
        // TODO: Support other card image types
    }

    // Announcement title
    let cardTitle = document.createElement('div');
    cardTitle.innerHTML = jsonContent.title;
    card.appendChild(cardTitle);

    let tempDiv = document.createElement('div');
    tempDiv.appendChild(card);
    return tempDiv;
}

function _renderLoopComponent(attachment) {
    let contentJson;
    let replaceString = null;
    try {
        contentJson = JSON.parse(attachment.content);
    } catch (error) {
        console.log(error);
    }
    if (contentJson.componentUrl) {
        let doOpenUrl = '"javascript:doOpenUrl(\'' + contentJson.componentUrl + '\')"';
        replaceString = 'Loop component: <a id="' + attachment.id + '" href=' + doOpenUrl
                + '>' + contentJson.componentUrl + '</a>';
    }
    return replaceString;
}
/**
 * Check and apply the highlight container style
 */
function _checkHighlightContainer() {
    // TODO: Use highlight container box
}

/**
 * Add user icon
 */
function _insertAvatarIcon(parentDiv, message) {
    let userDisplayName = null;
    if (message == null) {
        return;
    }

    if (message.from) {
        if (message.from.user) {
            userDisplayName = _getDisplayName(message.from.user);
        } else if (message.from.application) {
            userDisplayName = message.from.application.displayName;
        } else if (message.from.device) {
            userDisplayName = message.from.device.displayName;
        }
    }

    if (userDisplayName) {
        _insertUserNameIcon(parentDiv, userDisplayName);
    } else {
        let iconSource = './lib/images/default_avatar.png';
        iconSource = _getResource(iconSource);
        _insertUserIcon(parentDiv, iconSource);
    }
}

function _insertUserIcon(parentDiv, iconSource) {
    if (!iconSource) {
        return;
    }
    let iconDiv = document.createElement('div');
    let iconImg = document.createElement('img');

    iconDiv.className = 'img-cont-msg';
    iconImg.src = iconSource;
    iconImg.className = 'rounded-circle user-img-msg';

    iconDiv.appendChild(iconImg);
    parentDiv.appendChild(iconDiv);
}

function _insertUserNameIcon(parentDiv, userName) {
    if (!userName) {
        return;
    }
    let iconDiv = document.createElement('div');
    iconDiv.className = 'circle-avatar';
    const words = userName.toUpperCase().split(' ');
    let initial = words[0][0];
    if (words.length > 1) {
        initial = initial + words[1][0];
    } else {
        initial = initial + words[0][1];
    }
    iconDiv.textContent = initial;
    const iconColor = _stringToColour(userName);
    iconDiv.style.backgroundColor = iconColor;
    iconDiv.style.color = _invertColor(iconColor, true);
    parentDiv.appendChild(iconDiv);
}

/**
 * Add importance/mention icon
 */
function _insertImportanceIcon(parentDiv, isImportant, isUrgent, hasMentions, attachments) {
    let containerIcon = document.createElement('div');
    let hasAnnouncements = false;

    if (attachments && attachments.length > 0) {
        for (let item of attachments) {
            if (item.contentType === 'application/vnd.microsoft.teams.messaging-announcementBanner') {
                hasAnnouncements = true;
                break;
            }
        }
    }

    if (hasAnnouncements) {
        containerIcon.className = 'announcement-icon';
    } else if (hasMentions) {
        containerIcon.className = 'mention-icon';
    } else if (isImportant) {
        containerIcon.className = 'important-icon';
    } else if (isUrgent) {
        containerIcon.className = 'urgent-icon';
    }
    parentDiv.appendChild(containerIcon);
}

function _insertSubject(parentDiv, subject) {
    let subjectDiv = document.createElement('div');
    subjectDiv.className = 'subject';
    subjectDiv.innerHTML = subject;
    parentDiv.appendChild(subjectDiv);
}

function _insertImportanceTextTag(parentDiv, isImportant, isUrgent) {
    if (!isImportant && !isUrgent) {
        return;
    }
    let importanceText = document.createElement('div');
    importanceText.className = 'important-urgent-text';
    importanceText.innerHTML = ' ' + (isImportant ? LANG_IMPORTANT : LANG_URGENT) + '! ';
    parentDiv.appendChild(importanceText);
}

/**
 * Check if the message is sent from backup user
 */
function _isSelfMessage(message) {
    if (message == null || message.from == null) {
        return false;
    }
    var sentFrom = null;
    if (message.from.application != null) {
        sentFrom = message.from.application;
    } else if (message.from.device != null) {
        sentFrom = message.from.device;
    } else if (message.from.user != null) {
        sentFrom = message.from.user;
    }
    if (sentFrom != null && me != null) {
        return sentFrom.id == me.id;
    }
    return false;
}

/**
 * Get date value only
 */
function _getDateOnly(dateTimeString) {
    let date = new Date(dateTimeString);
    date.setHours(0,0,0,0);
    return date;
}

/**
 * Format date only for date line
 */
function _formatDate(date) {
    return date.toLocaleDateString(LOCALE, DATE_OPTIONS);
}

/**
 * Format date time for display in message
 */
function _formatDateTime(dateTime) {
    return dateTime.getDate() + '/' + (dateTime.getMonth() + 1)
        + '/' + dateTime.getFullYear()
        + ' ' + dateTime.toLocaleTimeString(LOCALE, TIME_OPTIONS);
}

function _stringToColour(str) {
    var hash = 0;
    for (var i = 0; i < str.length; i++) {
        hash = str.charCodeAt(i) + ((hash << 5) - hash);
    }
    var colour = '#';
    for (var i = 0; i < 3; i++) {
        var value = (hash >> (i * 8)) & 0xFF;
        colour += ('00' + value.toString(16)).substr(-2);
    }
    return colour;
}

function _invertColor(hex, bw) {
    if (hex.indexOf('#') === 0) {
        hex = hex.slice(1);
    }
    // convert 3-digit hex to 6-digits.
    if (hex.length === 3) {
        hex = hex[0] + hex[0] + hex[1] + hex[1] + hex[2] + hex[2];
    }
    if (hex.length !== 6) {
        throw new Error('Invalid HEX color.');
    }
    var r = parseInt(hex.slice(0, 2), 16),
        g = parseInt(hex.slice(2, 4), 16),
        b = parseInt(hex.slice(4, 6), 16);
    if (bw) {
        return (r * 0.299 + g * 0.587 + b * 0.114) > 186
            ? '#000000'
            : '#FFFFFF';
    }
    // invert color components
    r = (255 - r).toString(16);
    g = (255 - g).toString(16);
    b = (255 - b).toString(16);
    // pad each with zeros and return
    return "#" + _padZero(r) + _padZero(g) + _padZero(b);
}

function _padZero(str, len) {
    len = len || 2;
    var zeros = new Array(len).join('0');
    return (zeros + str).slice(-len);
}

function _replaceHostedContents(content) {
    var tempContent = content;
    if (content && hostedContentsList) {
        let urls = tempContent.match(/\b((https?|file):\/\/|(www)\.)[-A-Z0-9+&@#\/%?=~_|$!:,.;]*[A-Z0-9+&@#\/%=~_|$]/ig);
        if (urls) {
            var matchedUrls = {};
            for (let hostedContent of hostedContentsList) {
                for (let url of urls) {
                    if (url.includes(hostedContent.url)) {
                        matchedUrls[url] = hostedContent.localPath;
                    }
                }
            }
            for (let key in matchedUrls) {
                let url = key;
                let localPath = matchedUrls[key];
                if (localPath) {
                    tempContent = tempContent.replace(url, localPath);
                }
            }
        }
    }
    return tempContent;
}

function _checkHRef(msgDiv, href) {
    let tempHTML = msgDiv.innerHTML;
    let replaceString = 'href="javascript:doOpenUrl(\'' + href + '\')"';
    let findString = 'href="' + href + '"';
    msgDiv.innerHTML = tempHTML.replace(findString, replaceString);
}